/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.project.management;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.URL;
import java.net.URLDecoder;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

import org.simantics.db.DatabaseUserAgent;
import org.simantics.db.Driver;
import org.simantics.db.Manager;
import org.simantics.db.exception.DatabaseException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ServerManagerFactory {
    private static final Logger LOGGER = LoggerFactory.getLogger(ServerManagerFactory.class);

    public static ServerManager create(String databaseDriverId, String address) throws IOException, DatabaseException {
        Driver driver = Manager.getDriver(databaseDriverId);
        if (driver == null)
            throw new IllegalArgumentException("Database driver with ID " + databaseDriverId + " could not be found!");
        LOGGER.info("ServerManagerFactory.create called with id {}, driver is {}.", databaseDriverId, driver);
        DatabaseUserAgent agent = Manager.getUserAgent(databaseDriverId);
        if (agent != null)
            driver.setDatabaseUserAgent(address, agent);
        return new ServerManager(driver);
    }
	/**
	 * Create a server manager in OSGi platform.
	 *
	 * @return
	 * @throws IOException
	 * @throws ClassNotFoundException
	 * @throws RuntimeException unexpected problem with OSGi environment
	 */
	public static ServerManager createOSGI() throws IOException, RuntimeException {
		try {
			ClassLoader cl = ServerManager.class.getClassLoader();

			// Object bundle = Platform.getBundle("org.simantics.db.build");
			Class<?> Platform_class = cl.loadClass("org.eclipse.core.runtime.Platform");
			Method Platform_getBundle = Platform_class.getMethod("getBundle", String.class);
			Object bundle = Platform_getBundle.invoke(null, "org.simantics.db.build");
			if (bundle==null) throw new RuntimeException("Bundle org.simantics.db.build not found.");

			// URL db_build_url = bundle.getEntry("/");
			Class<?> Bundle_class = cl.loadClass("org.osgi.framework.Bundle");
			Method Bundle_getEntry = Bundle_class.getMethod("getEntry", String.class);
			URL db_build_url = (URL) Bundle_getEntry.invoke(bundle, "/");

	        // URL db_build_file_url = FileLocator.toFileURL( db_build_url );
			Class<?> FileLocator_class = cl.loadClass("org.eclipse.core.runtime.FileLocator");
			Method FileLocator_toFileURL = FileLocator_class.getMethod("toFileURL", URL.class);
			URL db_build_file_url = (URL) FileLocator_toFileURL.invoke(null, db_build_url);

			@SuppressWarnings("unused")
            String buildFile = URLDecoder.decode(db_build_file_url.getPath(), "UTF-8");

//	        File db_build_file = new File( buildFile );
	        //return new ServerManager( db_build_file );
	        throw new RuntimeException("ServerManager.createOSGI not implemented.");
		} catch( ClassNotFoundException e ) {
			throw new RuntimeException(e);
		} catch (SecurityException e) {
			throw new RuntimeException(e);
		} catch (NoSuchMethodException e) {
			throw new RuntimeException(e);
		} catch (IllegalArgumentException e) {
			throw new RuntimeException(e);
		} catch (IllegalAccessException e) {
			throw new RuntimeException(e);
		} catch (InvocationTargetException e) {
			throw new RuntimeException(e.getCause());
		}
	}

	/**
	 * Create a server manager in an POJO environment.
	 *
	 * @return
	 * @throws IOException
	 */
	public static ServerManager createPOJO() throws IOException {
		String tempPath = System.getenv("tmp");
		if (tempPath==null) tempPath = "c:/temp/";
		File driverDir = new File(tempPath+"/core_drivers");
		if (!driverDir.exists()) {
			System.out.println("Extracting Core drivers to "+driverDir);
			driverDir.mkdirs();
			ServerManagerFactory.extractDrivers(driverDir);
		} else {
			System.out.println("Loading Core drivers from "+driverDir);
		}
		//return new ServerManager(driverDir);
        throw new RuntimeException("ServerManager.createPOJO not implemented.");
	}

    public static ServerManager createPOJO(File driverDir) throws IOException {
        //return new ServerManager(driverDir);
        throw new RuntimeException("ServerManager.createPOJO not implemented.");
    }

    static ServerManager cached;

    public static boolean isOSGi() {
    	try {
    		ServerManager.class.getClassLoader().loadClass("org.osgi.framework.Bundle");
    		return true;
    	} catch (ClassNotFoundException e) {
    		return false;
    	}
    }

    public synchronized static ServerManager getServerManager() {
    	if (cached!=null) return cached;
    	try {
    		try {
    			return createOSGI();
    		} catch(RuntimeException e) {
    			return createPOJO();
    		}
    	} catch (IOException e) {
    		throw new RuntimeException(e);
    	}
    }


    /**
     * Extracts drivers files to a location.
     *
     * @param path location where to extract application files
     * @throws IOException
     */
	public static void extractDrivers(File path)
	throws IOException
	{
		// Extract org.simantics.db.build.zip to workspace
		InputStream is = ServerManager.class.getResource("org.simantics.db.build.zip").openStream();
		try {
			extractZip(is, path);
		} finally {
			is.close();
		}
	}

    /**
     * Extract a zip file into a directory
     *
     * @param zipInput
     * @param dst directory
     * @throws IOException
     */
    private static void extractZip(InputStream zipInput, File dst) throws IOException {
        byte[] buf = new byte[8192];
        ZipInputStream zis = new ZipInputStream(zipInput);
        ZipEntry entry;

        entry = zis.getNextEntry();
        while (entry != null) {
            // for each entry to be extracted
            String name = entry.getName();
            LOGGER.debug("Extracting "+name);
            File file = new File(dst, name);

            if (entry.isDirectory())
            {
                if ( !file.exists() ) file.mkdirs();
            } else {
                File parent = file.getParentFile();
                if (!parent.exists()) parent.mkdirs();
                if (!file.exists()) file.createNewFile();

                FileOutputStream fileoutputstream = new FileOutputStream(file);
                try {
                    int n = 0;
                    while ((n = zis.read(buf, 0, buf.length)) > -1)
                        fileoutputstream.write(buf, 0, n);
                } finally {
                    fileoutputstream.close();
                }
            }

            zis.closeEntry();
            entry = zis.getNextEntry();
        }// while

        zis.close();
    }


}

