/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.project.features;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Collection;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExecutableExtension;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.VirtualGraph;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.CommentMetadata;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.AssumptionException;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.ResourceNotFoundException;
import org.simantics.db.service.VirtualGraphSupport;
import org.simantics.layer0.Layer0;
import org.simantics.project.exception.ProjectException;
import org.simantics.project.ontology.ProjectResource;
import org.simantics.scl.reflection.OntologyVersions;

/**
 * A project feature for validating that a set of namespaces (URIs) are
 * reachable in the database for which this feature is configured.
 * 
 * The URIs are described as class arguments in the extension class spec (after
 * ':' character).
 * 
 * @author Tuukka Lehtonen
 */
public class DependencyValidationFeature extends AbstractProjectFeature implements IExecutableExtension {

    private String   virtualGraphId;
    private String[] uris = {};

    public DependencyValidationFeature() {
    }

    public DependencyValidationFeature(String virtualGraphId, String[] uris) {
        this.virtualGraphId = virtualGraphId;
        this.uris = uris;
    }

    @Override
    public void setInitializationData(IConfigurationElement config, String propertyName, Object data)
    throws CoreException {
        if (data instanceof String) {
            // Expecting comma-separated list of URIs in the argument.
            String[] uris = ((String) data).split(",");
            if (uris.length > 0) {
                for(int i=0;i<uris.length;i++) uris[i] = OntologyVersions.getInstance().currentVersion(uris[i]);
                // Validate the input data to contain valid URIs
                for (String uri : uris) {
                    try {
                        new URI(uri);
                    } catch (URISyntaxException e) {
                        throw new CoreException(new Status(IStatus.ERROR, config.getContributor().getName(), "extension configuration element executable extension attribute '" + propertyName + "' argument contains invalid URIs. See cause for details.", e));
                    }
                }

                this.uris = uris;
            }
        }
    }

    @Override
    public void configure() throws ProjectException {
        try {
            Session s = getSession();
            VirtualGraph vg = null;
            if (virtualGraphId != null) {
                VirtualGraphSupport support = s.getService(VirtualGraphSupport.class);
                vg = support.getWorkspacePersistent(virtualGraphId);
            }
            getSession().syncRequest(new WriteRequest(vg) {
                @Override
                public void perform(WriteGraph graph) throws DatabaseException {
                    configure(graph);
                    graph.addMetadata(graph.getMetadata(CommentMetadata.class).add("Configured by Dependency Validation Feature."));
                }
            });
        } catch (DatabaseException e) {
            throw new ProjectException(e.getMessage(), e);
        }
    }

    protected void configure(WriteGraph graph) throws DatabaseException {
        Collection<String> nss = new ArrayList<String>();
        Collection<String> notFound = new ArrayList<String>();

        Resource project = getProject().get();
        String projectName = NameUtils.getSafeName(graph, project);

        ProjectResource PROJ = ProjectResource.getInstance(graph);

        ArrayList<Resource> resourcesToLinkToProject = new ArrayList<Resource>();

        for (String uri : uris) {
            // This will fail if the extension-specified URI does not exist
            Resource namespaceRequirement = null;
            try {
                namespaceRequirement = graph.getResource(uri);
                resourcesToLinkToProject.add(namespaceRequirement);
            } catch (ResourceNotFoundException e) {
                notFound.add(uri);
                continue;
            }

            for (Resource nsp : graph.getObjects(namespaceRequirement, PROJ.RequiresNamespace)) {
                String ns = graph.getValue(nsp);
                nss.add(ns);
            }

            for (String ns : nss) {
                try {
                    // This will fail if the namespace is not found.
                    graph.getResource(ns);
                } catch (ResourceNotFoundException e) {
                    notFound.add(ns);
                }
            }
        }

        if (!notFound.isEmpty()) {
            StringBuilder sb = new StringBuilder();
            sb.append("Failed to locate the following namespaces required by project '");
            sb.append(projectName);
            sb.append("':\n");
            for (String nf : notFound) {
                sb.append("\t");
                sb.append(nf);
                sb.append("\n");
            }
            throw new AssumptionException(sb.toString());
        }

        // Ensure that the namespace requirements are linked to the project to
        // make them discoverable by database queries.
        linkTo(graph, project, resourcesToLinkToProject);
    }

    protected void linkTo(WriteGraph graph, Resource target, ArrayList<Resource> resourcesToLink) throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(graph);
        for (Resource resource : resourcesToLink) {
            if (!graph.hasStatement(target, L0.IsLinkedTo, resource))
                graph.claim(target, L0.IsLinkedTo, resource);
        }
    }

    @Override
    public void deconfigure() throws ProjectException {
    }

}
