/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.ui.property;

import java.lang.reflect.Array;
import java.util.Arrays;

import org.simantics.db.layer0.util.PrimitiveValueParser;
import org.simantics.utils.datastructures.slice.ValueRange;

/**
 * @author Tuukka Lehtonen
 */
public class ValueUtil {

    /**
     * @param object
     * @param range
     * @return
     * @throws ArrayIndexOutOfBoundsException if the specified index range is
     *         out of bounds for the specified object if it is an array.
     * @throws IllegalArgumentException if the specified value is not an array
     */
    public static String arrayValueToString(Object object, ValueRange range) {
        if (object == null)
            return "";

        Class<?> clazz = object.getClass();
        if (!clazz.isArray())
            throw new IllegalArgumentException("object is not an array: " + clazz.getCanonicalName());

        int length = Array.getLength(object);
        assert range != null;
        if (range.start() >= length || range.end() > length)
            throw new ArrayIndexOutOfBoundsException(range + " out of array bounds [0," + length + "]");

        StringBuilder b = new StringBuilder();

        boolean first = true;
        for (int i = range.start(); i < range.end(); ++i) {
            if (!first) {
                b.append(",");
            }
            first = false;
            b.append(Array.get(object, i));
        }
        return b.toString();
    }

    public static String valueToString(Object object) {
        if (object == null)
            return "";

        if (object instanceof String[]) {
            String[] value = (String[]) object;
            if (value.length == 1)
                return value[0];
            else
                return Arrays.toString(value);
        } else if (object instanceof int[]) {
            int[] value = (int[]) object;
            if (value.length == 1)
                return String.valueOf(value[0]);
            else {
                StringBuffer buffer = new StringBuffer();
                if (value.length > 0) {
                    buffer.append(value[0]);
                    for (int i = 1; i < value.length; ++i) {
                        buffer.append(", ");
                        buffer.append(value[i]);
                    }
                }
                return buffer.toString();
            }
        } else if (object instanceof boolean[]) {
            boolean[] value = (boolean[]) object;
            if (value.length == 1)
                return String.valueOf(value[0]);
            else
                return Arrays.toString(value);
        } else if (object instanceof double[]) {
            double[] value = (double[]) object;
            if (value.length == 1)
                return String.valueOf(value[0]);
            else {
                StringBuffer buffer = new StringBuffer();
                if(value.length > 0) {
                    buffer.append(value[0]);
                    for(int i=1;i<value.length;++i) {
                        buffer.append(", ");
                        buffer.append(value[i]);
                    }
                }
                return buffer.toString();
            }
        } else if (object instanceof float[]) {
            float[] value = (float[]) object;
            if (value.length == 1)
                return String.valueOf(value[0]);
            else {
                StringBuffer buffer = new StringBuffer();
                if(value.length > 0) {
                    buffer.append(value[0]);
                    for(int i=1;i<value.length;++i) {
                        buffer.append(", ");
                        buffer.append(value[i]);
                    }
                }
                return buffer.toString();
            }
        } else if (object instanceof long[]) {
            long[] value = (long[]) object;
            if (value.length == 1)
                return String.valueOf(value[0]);
            else
                return Arrays.toString(value);
        } else if (object instanceof byte[]) {
            byte[] value = (byte[]) object;
            if (value.length == 1)
                return String.valueOf(value[0]);
            else
                return Arrays.toString(value);
        }
        return object.toString();
    }

    /**
     * @param valueClass
     * @param label
     * @return
     * @throws IllegalArgumentException if parsing of the specified label into
     *         the specified Class fails
     */
    public static Object parseLabelToObject(Class<?> valueClass, String label) {
        if (valueClass == boolean[].class) {
            return PrimitiveValueParser.parseBooleanArray(label);
        } else if (valueClass == int[].class) {
            return PrimitiveValueParser.parseIntArray(label);
        } else if (valueClass == long[].class) {
            return PrimitiveValueParser.parseLongArray(label);
        } else if (valueClass == float[].class) {
            return PrimitiveValueParser.parseFloatArray(label);
        } else if (valueClass == double[].class) {
            return PrimitiveValueParser.parseDoubleArray(label);
        } else if (valueClass == String[].class) {
            return new String[] { label };
        } else if (valueClass == boolean.class || valueClass == Boolean.class) {
            return PrimitiveValueParser.parseBoolean(label);
        } else if (valueClass == int.class || valueClass == Integer.class) {
            return PrimitiveValueParser.parseInt(label);
        } else if (valueClass == long.class || valueClass == Long.class) {
            return PrimitiveValueParser.parseLong(label);
        } else if (valueClass == float.class || valueClass == Float.class) {
            return PrimitiveValueParser.parseFloat(label);
        } else if (valueClass == double.class || valueClass == Double.class) {
            return PrimitiveValueParser.parseDouble(label);
        } else if (valueClass == Byte.class || valueClass == Byte.class) {
            return PrimitiveValueParser.parseByte(label);
        } else if (valueClass == String.class) {
            return label;
        }
        return null;
    }

}
