/*******************************************************************************
 * Copyright (c) 2007, 2012 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.ui.property;

import org.eclipse.core.expressions.PropertyTester;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.simantics.DatabaseJob;
import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.RequestProcessor;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.common.request.UniqueRead;
import org.simantics.db.common.utils.RequestUtil;
import org.simantics.db.exception.DatabaseException;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.modeling.ModelingResources;
import org.simantics.ui.SimanticsUI;
import org.simantics.ui.workbench.IResourceEditorInput;
import org.simantics.utils.ui.ErrorLogger;

/**
 * @author Tuukka Lehtonen
 */
public class TypicalPropertyTester extends PropertyTester {


    /**
     * Tests if the received object is considered deletable.
     */
    private static final String IS_TYPICAL_MASTER_EDITOR = "isMasterEditor";

    /**
     * Tests if the received object is considered modifiable.
     */
    private static final String IS_TYPICAL_INSTANCE_EDITOR = "isInstanceEditor";

    @Override
    public boolean test(final Object receiver, final String property, final Object[] args, final Object expectedValue) {
        //System.out.println("TEST: " + receiver + ", " + property + ", " + Arrays.toString(args) + ", " + expectedValue);

        try {
            Session session = Simantics.peekSession();
            if (session == null)
                return false;

            if (!(receiver instanceof IEditorPart))
                return false;
            IEditorPart editor = (IEditorPart) receiver;
            IEditorInput in = editor.getEditorInput();
            if (!(in instanceof IResourceEditorInput))
                return false;
            IResourceEditorInput input = (IResourceEditorInput) in;
            final Resource inputResource = input.getResource();

            if (DatabaseJob.inProgress()) {
                // See Apros issue #9115
                // Return true because it is often possible that the database
                // will be busy when these properties are tested. In such cases
                // the handlers/menu contributions using these tests would
                // become disabled unless we return true here. It is up to the
                // handlers to also make sure that their input is valid.
                return true;
            }

            if (IS_TYPICAL_MASTER_EDITOR.equals(property)) {
                return isTypicalMasterEditor(session, inputResource);
            } else if (IS_TYPICAL_INSTANCE_EDITOR.equals(property)) {
                return isTypicalInstanceEditor(session, inputResource);
            }
        } catch (DatabaseException | InterruptedException e) {
            ErrorLogger.defaultLogError(e);
        }

        return false;
    }

    public static boolean isTypicalMasterEditor(RequestProcessor processor, final Resource editorInputResource) throws DatabaseException, InterruptedException {
        return RequestUtil.trySyncRequest(
                Simantics.getSession(),
                SimanticsUI.UI_THREAD_REQUEST_START_TIMEOUT,
                SimanticsUI.UI_THREAD_REQUEST_EXECUTION_TIMEOUT,
                false,
                new UniqueRead<Boolean>() {
            @Override
            public Boolean perform(ReadGraph graph) throws DatabaseException {
                ModelingResources MOD = ModelingResources.getInstance(graph);
                Resource composite = graph.getPossibleObject(editorInputResource, MOD.DiagramToComposite);
                return composite != null
                        && graph.isInstanceOf(composite, MOD.MasterTypicalCompositeType);
            }
        });
    }

    public static boolean isTypicalInstanceEditor(RequestProcessor processor, final Resource editorInputResource) throws DatabaseException, InterruptedException {
        return RequestUtil.trySyncRequest(
                Simantics.getSession(),
                SimanticsUI.UI_THREAD_REQUEST_START_TIMEOUT,
                SimanticsUI.UI_THREAD_REQUEST_EXECUTION_TIMEOUT,
                false,
                new UniqueRead<Boolean>() {
            @Override
            public Boolean perform(ReadGraph graph) throws DatabaseException {
                DiagramResource DIA = DiagramResource.getInstance(graph);
                ModelingResources MOD = ModelingResources.getInstance(graph);
                return graph.isInstanceOf(editorInputResource, DIA.Diagram)
                        && graph.hasStatement(editorInputResource, MOD.HasDiagramSource);
            }
        });
    }

}
