/*******************************************************************************
 * Copyright (c) 2007, 2012 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.ui.property;

import org.eclipse.core.expressions.PropertyTester;
import org.simantics.DatabaseJob;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.common.request.UniqueRead;
import org.simantics.db.common.utils.RequestUtil;
import org.simantics.db.exception.DatabaseException;
import org.simantics.layer0.Layer0;
import org.simantics.structural.stubs.StructuralResource2;
import org.simantics.ui.SimanticsUI;
import org.simantics.ui.utils.ResourceAdaptionUtils;

/**
 * A JFace property tester extension for the Eclipse command framework that is
 * meant for working with Simantics database {@link Resource} instances.
 * 
 * <p>
 * This tester expects to receive IStructuredSelections that contain Resource
 * instances.
 * 
 * <p>
 * It supports testing of the following properties:
 * <ul>
 * <li>partOf - </li>
 * </ul>
 * 
 * @author Tuukka Lehtonen
 */
public class ComponentTypePropertyTester extends PropertyTester {

    /**
     * Tests to see if the input Resource is part of a structural component
     * type.
     */
    private static final String PART_OF = "partOf";

    @Override
    public boolean test(final Object receiver, final String property, final Object[] args, final Object expectedValue) {
        final Resource resource = ResourceAdaptionUtils.toSingleResource(receiver);
        if (resource == null)
            return false;

        Session session = SimanticsUI.peekSession();
        if (session == null)
            return false;

        if (DatabaseJob.inProgress())
            return false;

        try {
            return RequestUtil.trySyncRequest(
                    session,
                    SimanticsUI.UI_THREAD_REQUEST_START_TIMEOUT,
                    SimanticsUI.UI_THREAD_REQUEST_EXECUTION_TIMEOUT,
                    false,
                    new UniqueRead<Boolean>() {
                @Override
                public Boolean perform(ReadGraph g) throws DatabaseException {
                    return Boolean.valueOf(doTest(g, resource, property, args, expectedValue));
                }
            });
        } catch (DatabaseException | InterruptedException e) {
            // Purposefully not logging these exceptions, there might be way too
            // many even under normal circumstances.
            // TODO: add debug tracing options controlling the printing of these exceptions
            return false;
        }
    }

    private boolean doTest(ReadGraph graph, Resource resource, String property, Object[] args, Object expectedValue) throws DatabaseException {
        if (PART_OF.equals(property)) {
            boolean expected = parseBoolean(expectedValue, true);
            boolean is = partOfComponentType(graph, resource) != null;
            return is == expected;
        }
        return false;
    }

    private Resource partOfComponentType(ReadGraph graph, Resource resource) throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(graph);
        StructuralResource2 STR = StructuralResource2.getInstance(graph);
        Resource componentType = null;
        for (Resource curComponent = resource; true;) {
            componentType = graph.getPossibleObject(curComponent, STR.Defines);
            if (componentType != null) {
                return componentType;
            }
            curComponent = graph.getPossibleObject(curComponent, L0.PartOf);
            if (curComponent == null)
                break;  
        }
        return null;
    }

    boolean parseBoolean(Object value, boolean defaultValue) {
        if (value instanceof Boolean)
            return (Boolean) value;
        if (value instanceof String)
            return Boolean.parseBoolean((String) value);
        return defaultValue;
    }

}
