/*******************************************************************************
 * Copyright (c) 2014 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.ui.function;

import java.io.File;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.simantics.Simantics;
import org.simantics.browsing.ui.BuiltinKeys;
import org.simantics.browsing.ui.ExplorerState;
import org.simantics.browsing.ui.NodeContext;
import org.simantics.browsing.ui.NodeContext.ConstantKey;
import org.simantics.browsing.ui.common.NodeContextBuilder;
import org.simantics.browsing.ui.common.NodeContextBuilder.MapNodeContext;
import org.simantics.browsing.ui.model.actions.ActionBrowseContext;
import org.simantics.browsing.ui.model.browsecontexts.BrowseContext;
import org.simantics.browsing.ui.model.browsecontexts.BrowseContexts;
import org.simantics.browsing.ui.model.nodetypes.EntityNodeType;
import org.simantics.browsing.ui.model.nodetypes.NodeType;
import org.simantics.browsing.ui.model.nodetypes.SpecialNodeType;
import org.simantics.browsing.ui.swt.GraphExplorerStateBean;
import org.simantics.browsing.ui.swt.GraphExplorerStateNodeBean;
import org.simantics.browsing.ui.swt.IdentifiedStatePersistor;
import org.simantics.browsing.ui.swt.NodeContextValueBean;
import org.simantics.browsing.ui.swt.StringArrayBean;
import org.simantics.browsing.ui.swt.StringBean;
import org.simantics.databoard.Bindings;
import org.simantics.databoard.binding.impl.ArrayListBinding;
import org.simantics.databoard.util.Bean;
import org.simantics.databoard.util.StringUtil;
import org.simantics.db.Resource;

/**
 * @author Antti Villberg
 * @author Tuukka Lehtonen
 */
class StandardPersistor extends IdentifiedStatePersistor {

	public StandardPersistor(String configurationId) {
		super(configurationId);
	}

	@Override
	protected Path getMementoPath(File stateLocation, NodeContext root) {
		if (root != null) {
			Object input = root.getConstant(BuiltinKeys.INPUT);
			if (input instanceof Resource) {
				Resource r = (Resource) input;
				return stateLocation.toPath().resolve(StringUtil.escapeToFileName(id + "#" + r.getResourceId() + ".ge"));
			}
		}
		return null;
	}

	@Override
	protected GraphExplorerStateBean toStateBean(ExplorerState state, NodeContext root) {
		Object input = root.getConstant(BuiltinKeys.INPUT);
		if (input instanceof Resource)
			return super.toStateBean(state, root);
		return null;
	}

	@Override
	protected GraphExplorerStateNodeBean[] toNodeBeans(NodeContext[] contexts) {
		ArrayList<GraphExplorerStateNodeBean> result = new ArrayList<GraphExplorerStateNodeBean>(contexts.length);
		for (NodeContext node : contexts) {
			GraphExplorerStateNodeBean bean = buildNodeBean(node);
			if (bean != null) {
				result.add(bean);
			}
		}
		return result.toArray(GraphExplorerStateNodeBean.NONE);
	}

	@Override
	protected NodeContext[] toNodeContexts(GraphExplorerStateNodeBean[] beans) throws Exception {
		ArrayList<NodeContext> result = new ArrayList<NodeContext>(beans.length);
		for (GraphExplorerStateNodeBean bean : beans) {
			NodeContext node = bean != null ? buildNodeContext(bean) : null;
			if (node != null) {
				result.add(node);
			}
		}
		return result.toArray(NodeContext.NONE);
	}

	protected NodeContext buildNodeContext(GraphExplorerStateNodeBean node) throws Exception {
		ArrayList<Object> keysAndValues = new ArrayList<Object>(node.map.size());
		for(Map.Entry<String, Bean> entry : node.map.entrySet()) {
			ConstantKey<?> key = decodeKey(entry.getKey());
			Object value = decodeValue(key, entry.getValue());
			keysAndValues.add(key);
			keysAndValues.add(value);
		}
		return NodeContextBuilder.buildWithData(keysAndValues.toArray());
	}

	protected GraphExplorerStateNodeBean buildNodeBean(NodeContext nodeContext) {
		if (nodeContext instanceof MapNodeContext) {
			MapNodeContext ctx = (MapNodeContext) nodeContext;
			GraphExplorerStateNodeBean node = new GraphExplorerStateNodeBean();
			for (ConstantKey<?> key : ctx.getKeys()) {
				Object value = ctx.getConstant(key);
				if (value instanceof Resource) {
					node.map.put(key.getClass().getName(), new NodeContextValueBean((Resource) value));
				} else if (value instanceof EntityNodeType) {
					node.map.put(key.getClass().getName(), new NodeContextValueBean((EntityNodeType) value));
				} else if (value instanceof SpecialNodeType) {
					node.map.put(key.getClass().getName(), new NodeContextValueBean((SpecialNodeType) value));
				} else if (key == BuiltinKeys.UI_CONTEXT) {
					node.map.put(key.getClass().getName(), new StringBean((String)value));
				} else if (key == BuiltinKeys.BROWSE_CONTEXT) {
					String[] uris = value != null ? ((BrowseContext) value).getURIs() : null;
					node.map.put(key.getClass().getName(), new StringArrayBean(uris));
				} else if (key == BuiltinKeys.ACTION_BROWSE_CONTEXT) {
					String[] uris = value != null ? ((ActionBrowseContext) value).getURIs() : null;
					node.map.put(key.getClass().getName(), new StringArrayBean(uris));
				}
			}
			return node;
		}
		return null;
	}

	private ConstantKey<?> decodeKey(String name) {
		if(BuiltinKeys.INPUT.getClass().getName().equals(name)) return BuiltinKeys.INPUT;
		else if(BuiltinKeys.UI_CONTEXT.getClass().getName().equals(name)) return BuiltinKeys.UI_CONTEXT;
		else if(BuiltinKeys.BROWSE_CONTEXT.getClass().getName().equals(name)) return BuiltinKeys.BROWSE_CONTEXT;
		else if(BuiltinKeys.ACTION_BROWSE_CONTEXT.getClass().getName().equals(name)) return BuiltinKeys.ACTION_BROWSE_CONTEXT;
		else if(NodeType.TYPE.getClass().getName().equals(name)) return NodeType.TYPE;
		else throw new IllegalArgumentException(name);
	}

	private Object decodeValue(ConstantKey<?> key, Bean bean) throws Exception {
		
		if (key == BuiltinKeys.BROWSE_CONTEXT) {
			String[] uris = (String[]) bean.getField("strings");
			return uris == null ? null : BrowseContexts.toBrowseContext(Simantics.getSession(), uris);
		} else if (key == BuiltinKeys.ACTION_BROWSE_CONTEXT) {
			String[] uris = (String[]) bean.getField("strings");
			return uris == null ? null : BrowseContexts.toActionBrowseContext(Simantics.getSession(), uris);
		} else if (key == BuiltinKeys.UI_CONTEXT) {
			return bean.getField("string");
		}

		String name = (String)bean.getField("name");
		if(Resource.class.getName().equals(name)) {
			return bean.getField("resource", Bindings.getBinding(Resource.class));
		} else if(EntityNodeType.class.getName().equals(name)) {
			@SuppressWarnings("unchecked")
			List<Resource> entityType = (List<Resource>)bean.getField("resources", 
					new ArrayListBinding(Bindings.getBinding(Resource.class)));
			return EntityNodeType.create(entityType);
		} else if(SpecialNodeType.class.getName().equals(name)) {
			Resource resource = (Resource)bean.getField("resource", Bindings.getBinding(Resource.class));
			String contentType = (String)bean.getField("className");
			Class<?> clazz = getClass().getClassLoader().loadClass(contentType);
			return new SpecialNodeType(resource, clazz);
		}
		throw new IllegalArgumentException("key = " + key + ", bean = " + bean);
	}

}