/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.ui.diagramEditor;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.common.utils.ListUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.diagram.content.ConnectionUtil;
import org.simantics.diagram.flag.FlagUtil;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.issues.ontology.IssueResource;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ModelingResources;
import org.simantics.modeling.ui.Activator;
import org.simantics.ui.SimanticsUI;
import org.simantics.ui.workbench.editor.AbstractResourceEditorAdapter;

/**
 * @author Tuukka Lehtonen
 */
public class OpenDiagramFromIssue extends AbstractResourceEditorAdapter {

    private static final String EDITOR_ID = "org.simantics.modeling.ui.plainDiagramEditor";

    public OpenDiagramFromIssue() {
        super("Open Diagram Containing Referenced Element", Activator.COMPOSITE_ICON);
    }

    protected String getEditorId() {
        return EDITOR_ID;
    }

    @Override
    public boolean canHandle(ReadGraph g, Resource r) throws DatabaseException {
        return findElement(g, r) != null;
    }

    @Override
    public void openEditor(final Resource r) throws Exception {
        SimanticsUI.getSession().asyncRequest(new ReadRequest() {
            @Override
            public void run(ReadGraph g) throws DatabaseException {
            	
            	Layer0 L0 = Layer0.getInstance(g);
            	ModelingResources MOD = ModelingResources.getInstance(g);
            	
            	Resource element = findElement(g, r);
            	if(element == null) return;
            	
            	System.err.println("element " + g.getURI(element));
            	
            	Resource diagram = g.getSingleObject(element, L0.PartOf);
            	Resource composite = g.getSingleObject(diagram, MOD.DiagramToComposite);
            			
                OpenDiagramFromConfigurationAdapter.openEditor(g, composite, getEditorId(), Collections.<Object>singleton(element));
                
            }
        });
    }

    private static Resource findConfigurationForElement(ReadGraph graph, Resource element) throws DatabaseException {

    	Layer0 L0 = Layer0.getInstance(graph);
    	ModelingResources MOD = ModelingResources.getInstance(graph);

    	Resource diagram = graph.getPossibleObject(element, L0.PartOf);
    	if (diagram == null) return null;

    	return graph.getPossibleObject(diagram, MOD.DiagramToComposite);
    	
    }
    
    private static Resource findElement(ReadGraph graph, Resource issue) throws DatabaseException {
    	
    	IssueResource ISSUE = IssueResource.getInstance(graph);
    	DiagramResource DIA = DiagramResource.getInstance(graph);
    	
    	if(!graph.isInstanceOf(issue, ISSUE.Issue)) return null;
    	
    	Resource list = graph.getSingleObject(issue, ISSUE.Issue_HasContexts);
    	for(Resource context : ListUtils.toList(graph, list)) {
    		if(graph.isInstanceOf(context, DIA.Element) && !graph.isInstanceOf(context, DIA.Diagram)) {
    			return context;
    		}
    		
    	}
    	
    	return null;
    	
    }

    public static Collection<Object> findElementObjects(ReadGraph g, Resource module) throws DatabaseException {
        DiagramResource DIA = DiagramResource.getInstance(g);
        ModelingResources MOD = ModelingResources.getInstance(g);
        final Collection<Object> selectedObjects = new ArrayList<Object>();
        for (Resource element : g.getObjects(module, MOD.ComponentToElement)) {
            if (g.isInstanceOf(element, DIA.Flag) && FlagUtil.isExternal(g, element)) {
                // Use external flag primarily if one exists in the correspondences
                selectedObjects.clear();
                selectedObjects.add(element);
                break;
            } else if (g.isInstanceOf(element, DIA.RouteGraphConnection)) {
                selectedObjects.add(element);
            } else if (g.isInstanceOf(element, DIA.Connection)) {
                // Okay, we need to find a part of the connection
                ConnectionUtil cu = new ConnectionUtil(g);
                cu.gatherConnectionParts(element, selectedObjects);
            } else {
                selectedObjects.add(element);
            }
        }
        for(Resource element : g.getObjects(module, MOD.HasParentComponent_Inverse)) {
        	selectedObjects.add(element);
        }
        return selectedObjects;
    }    
    
}
