/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.ui.diagramEditor;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.ActionContributionItem;
import org.eclipse.jface.action.IContributionItem;
import org.eclipse.jface.action.ICoolBarManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.ToolBarContributionItem;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IPartListener2;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchPartReference;
import org.eclipse.ui.part.EditorActionBarContributor;
import org.simantics.g2d.canvas.Hints;
import org.simantics.g2d.canvas.ICanvasContext;
import org.simantics.g2d.canvas.IToolMode;
import org.simantics.modeling.ui.Activator;
import org.simantics.utils.datastructures.hints.HintListenerAdapter;
import org.simantics.utils.datastructures.hints.IHintContext.Key;
import org.simantics.utils.datastructures.hints.IHintListener;
import org.simantics.utils.datastructures.hints.IHintObservable;
import org.simantics.utils.threads.ThreadUtils;

/**
 * @author Tuukka Lehtonen
 */
public class DiagramViewerActionContributor extends EditorActionBarContributor {

    private static final boolean DEBUG = false;

    private IEditorPart    activePart;
    private Display        display;

    private ICanvasContext currentContext;

    ModeAction             pointerAction;
    ModeAction             connectAction;
    IContributionItem      pointerItem;
    IContributionItem      connectItem;
    protected IToolBarManager        mgr;

    IPartListener2 partListener = new IPartListener2() {
        @Override
        public void partVisible(IWorkbenchPartReference partRef) {
        }
        @Override
        public void partOpened(IWorkbenchPartReference partRef) {
        }
        @Override
        public void partInputChanged(IWorkbenchPartReference partRef) {
        }
        @Override
        public void partHidden(IWorkbenchPartReference partRef) {
        }
        @Override
        public void partDeactivated(IWorkbenchPartReference partRef) {
            IWorkbenchPart part = partRef.getPart(false);
            if (part == activePart) {
                setActiveEditor(null);
            }
        }
        @Override
        public void partClosed(IWorkbenchPartReference partRef) {
        }
        @Override
        public void partBroughtToTop(IWorkbenchPartReference partRef) {
        }
        @Override
        public void partActivated(IWorkbenchPartReference partRef) {
        }
    };

    @Override
    public void contributeToCoolBar(ICoolBarManager coolBarManager) {
        IContributionItem item = coolBarManager.find("org.simantics.modeling.ui.diagramtoolbar");
        if (item instanceof ToolBarContributionItem)
            mgr = ((ToolBarContributionItem) item).getToolBarManager();
        if (mgr == null)
            return;

        pointerAction = new ModeAction("org.simantics.modeling.ui.pointerMode", Hints.POINTERTOOL);
        pointerAction.setText("Pointer Mode");
        pointerAction.setImageDescriptor(Activator.POINTER_MODE);
        connectAction = new ModeAction("org.simantics.modeling.ui.connectMode", Hints.CONNECTTOOL);
        connectAction.setText("Connect Mode");
        connectAction.setImageDescriptor(Activator.CONNECT_MODE);

        pointerItem = new ActionContributionItem(pointerAction);
        connectItem = new ActionContributionItem(connectAction);

        mgr.appendToGroup("tool.additions", pointerItem);
        mgr.appendToGroup("tool.additions", connectItem);
        mgr.markDirty();

        getPage().addPartListener(partListener);
    }

    @Override
    public void dispose() {
        getPage().removePartListener(partListener);

        if (mgr != null) {
            mgr.remove(connectItem);
            mgr.remove(pointerItem);
            connectItem.dispose();
            pointerItem.dispose();
            mgr.markDirty();
            mgr.update(true);
            if (activePart != null) {
                activePart.getEditorSite().getActionBars().updateActionBars();
                activePart = null;
            }
        }
        setContext(null, null);
        super.dispose();
        setContext(null, null);
        activePart = null;
    }

    public void setActiveEditor(IEditorPart part) {
        if (DEBUG)
            System.out.println("setActiveEditor: " + part);
        if (part == activePart)
            return;

        this.activePart = part;
        if (part != null) {
            this.display = part.getSite().getShell().getDisplay();
        }

        ICanvasContext ctx = part != null ? (ICanvasContext) part.getAdapter(ICanvasContext.class) : null;
        setContext(part, ctx);
    }

    private void setContext(IEditorPart part, ICanvasContext context) {
        if (DEBUG)
            System.out.println("setContext: " + part + "\nNEW CONTEXT: " + context + "\nPREVIOUS CONTEXT: " + currentContext);

        ICanvasContext previous = currentContext;
        currentContext = context;
        if (previous != context) {
            if (previous != null && !previous.isDisposed())
                previous.getHintStack().removeKeyHintListener(Hints.KEY_TOOL, TOOL_LISTENER);
            if (context != null && !context.isDisposed())
                context.getHintStack().addKeyHintListener(Hints.KEY_TOOL, TOOL_LISTENER);
        }

        if (part != null && context != null && !context.isDisposed())
            updateActionBars(part, (IToolMode) context.getHintStack().getHint(Hints.KEY_TOOL));
        else
            updateActionBars(null, null);
    }

    private void updateActionBars(IEditorPart part, IToolMode toolMode) {
        if (DEBUG)
            System.out.println("updateActionBars: " + part + ", " + toolMode);
        updateToolMode(toolMode);
        if (part != null)
            part.getEditorSite().getActionBars().updateActionBars();
    }

    private void updateToolMode(IToolMode toolMode) {
        if (DEBUG)
            System.out.println("updateToolMode: " + toolMode);
        if (toolMode != null) {
            pointerAction.setEnabled(true);
            connectAction.setEnabled(true);
            if (toolMode == Hints.POINTERTOOL) {
                pointerAction.setChecked(true);
                connectAction.setChecked(false);
            } else if (toolMode == Hints.CONNECTTOOL) {
                pointerAction.setChecked(false);
                connectAction.setChecked(true);
            }
        } else {
            pointerAction.setEnabled(false);
            connectAction.setEnabled(false);
        }
    }

    IHintListener TOOL_LISTENER = new HintListenerAdapter() {
        @Override
        public void hintChanged(IHintObservable sender, Key key, Object oldValue, Object newValue) {
            final IToolMode mode = (IToolMode) newValue;
            final IEditorPart part = activePart;
            display.asyncExec(new Runnable() {
                @Override
                public void run() {
                    if (display.isDisposed())
                        return;
                    if (activePart != part)
                        return;
                    updateActionBars(part, mode);
                }
            });
        }
    };

    public class ModeAction extends Action {

        private IToolMode targetMode;

        public ModeAction(String id, IToolMode targetMode) {
            super(id, Action.AS_RADIO_BUTTON);
            this.targetMode = targetMode;
        }

        @Override
        public void run() {
            final ICanvasContext context = currentContext;
            if (context == null)
                return;

            final IEditorPart part = activePart;
            if (part == null)
                return;

            if (context.isDisposed())
                return;

            ThreadUtils.asyncExec(context.getThreadAccess(), new Runnable() {
                @Override
                public void run() {
                    if (context.isDisposed())
                        return;
                    IToolMode toolMode = context.getHintStack().getHint(Hints.KEY_TOOL);
                    if (!targetMode.equals(toolMode)) {
                        if (DEBUG)
                            System.out.println("changing tool mode to " + targetMode.getId());
                        context.getDefaultHintContext().setHint(Hints.KEY_TOOL, targetMode);
                    }
                }
            });
        }

    }

}
