/*******************************************************************************
 * Copyright (c) 2012 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.ui.diagram.monitor;

import org.simantics.common.format.Formatter;
import org.simantics.databoard.Bindings;
import org.simantics.datatypes.literal.DecimalFormatFormatter;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.BinaryRead;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.exception.VariableException;
import org.simantics.db.layer0.variable.RVI;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.layer0.variable.Variables;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.operation.Layer0X;

/**
 * @author Tuukka Lehtonen
 */
public class ResolveMonitorVariable extends BinaryRead<Resource, Resource, MonitorVariable> {

    public ResolveMonitorVariable(Resource runtimeDiagram, Resource monitorElement) {
        super(runtimeDiagram, monitorElement);
    }

    @Override
    public MonitorVariable perform(ReadGraph graph) throws DatabaseException {
        Boolean external = null;
        DiagramResource DIA = DiagramResource.getInstance(graph);

        Resource monitorComponent = graph.getPossibleObject(parameter2, DIA.HasMonitorComponent);
        if (monitorComponent == null)
            return null;

        String suffix = graph.getPossibleRelatedValue(parameter2, DIA.HasMonitorSuffix, Bindings.STRING);
        if (suffix == null)
            return null;

        String diagramVariable = graph.getPossibleRelatedValue(parameter, DIA.RuntimeDiagram_HasVariable);
        if (diagramVariable == null)
            return null;

        Variable var = Variables.getPossibleVariable(graph, diagramVariable);
        if (var == null)
            return null;

        Variable component = var.browsePossible(graph, monitorComponent);
        if (component == null) {
            // The monitored component is not in this diagram.
            component = Variables.getPossibleVariable(graph, monitorComponent);
            if (component == null)
                return null;

            String componentRVI = Variables.getPossibleRVI(graph, component);
            if (componentRVI == null)
                return null;

            Resource realization = Variables.getRealization(graph, var);
            String diagramRealizationURI = graph.getPossibleURI(realization);
            if (diagramRealizationURI == null)
                return null;

            String componentURI = diagramRealizationURI + componentRVI;
            var = Variables.getPossibleVariable(graph, componentURI);
            if (var == null)
                return null;

            external = Boolean.TRUE;
        } else {
            var = component;
        }

        var = var.browsePossible(graph, suffix);
        if (var == null)
            return null;

        Formatter formatter = graph.getPossibleRelatedAdapter(parameter2, DIA.HasFormatter, Formatter.class);
        if (formatter == null)
            formatter = new DecimalFormatFormatter();

        Layer0X L0X = Layer0X.getInstance(graph);
        String _expression = graph.getPossibleRelatedAdapter(parameter2, L0X.HasExpression, String.class);
        if(_expression == null) _expression  = "value";

        RVI rvi = null;
        try {
            rvi = Variables.getRVI2(graph, var);
        } catch (VariableException e) {
            // No RVI, which is fine for contents that do not originate from the model configuration.
        }

        return new MonitorVariable(monitorComponent, formatter, _expression, var, rvi, external);
    }

}