package org.simantics.modeling.ui.componentTypeEditor;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.Collections;
import java.util.List;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.operation.IRunnableContext;
import org.eclipse.jface.text.Document;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.Position;
import org.eclipse.jface.text.source.Annotation;
import org.eclipse.jface.text.source.AnnotationModel;
import org.eclipse.jface.text.source.IAnnotationModel;
import org.eclipse.ui.texteditor.AbstractDocumentProvider;
import org.simantics.Simantics;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.common.request.UniqueRead;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.util.Layer0Utils;
import org.simantics.layer0.Layer0;
import org.simantics.scl.compiler.errors.CompilationError;
import org.simantics.scl.compiler.errors.Locations;
import org.simantics.ui.workbench.ResourceEditorInput;
import org.simantics.utils.logging.TimeLogger;

public class PGraphEditorDocumentProvider extends AbstractDocumentProvider {

    protected Resource resource;
    protected String currentText;
    protected boolean errorHappened;

    protected AnnotationModel annotationModel = new AnnotationModel();

    @Override
    protected IDocument createDocument(Object element) throws CoreException {
        ResourceEditorInput input = (ResourceEditorInput)element;
        resource = input.getResource();
        try {
            return Simantics.getSession().syncRequest(new UniqueRead<Document>() {
                @Override
                public Document perform(ReadGraph graph) throws DatabaseException {
                    Layer0 L0 = Layer0.getInstance(graph);
                    currentText = graph.getRelatedValue(resource, L0.PGraph_definition, Bindings.STRING);
                    errorHappened = false;
                    return new Document(currentText != null ? currentText : "");
                }
            });
        } catch (DatabaseException e) {
            StringWriter sw = new StringWriter();
            PrintWriter pw = new PrintWriter(sw);
            e.printStackTrace(pw);
            errorHappened = true;
            return new Document(sw.toString());
        }
    }

    protected void updateAnnotations() {
        Simantics.getSession().asyncRequest(new ReadRequest() {
            @Override
            public void run(ReadGraph graph) throws DatabaseException {
                setAnnotations(Collections.emptyList());
            }
        });
    }

    protected void setAnnotations(List<CompilationError> errors) {
        synchronized(annotationModel.getLockObject()) {
            annotationModel.removeAllAnnotations();
            for(CompilationError error : errors) {
                Annotation annotation = new Annotation("org.eclipse.ui.workbench.texteditor.error", true,
                        error.description);
                int begin = Locations.beginOf(error.location);
                int end = Locations.endOf(error.location);
                Position position = new Position(begin, end - begin);
                annotationModel.addAnnotation(annotation, position);
            }
        }
    }

    boolean annotationsInitialized = false;

    @Override
    protected IAnnotationModel createAnnotationModel(Object element)
            throws CoreException {
        if(!annotationsInitialized) {
            updateAnnotations();
            annotationsInitialized = true;
        }
        return annotationModel;
    }

    @Override
    protected void doSaveDocument(IProgressMonitor monitor, Object element,
            IDocument document, boolean overwrite) throws CoreException {
        TimeLogger.resetTimeAndLog("PGraphEditorDocumentProvider.doSaveDocument");
        currentText = document.get();
        Simantics.getSession().asyncRequest(new WriteRequest() {
            @Override
            public void perform(WriteGraph graph) throws DatabaseException {
                graph.markUndoPoint();
                Layer0 L0 = Layer0.getInstance(graph);
                graph.claimLiteral(resource, L0.PGraph_definition, currentText, Bindings.STRING);
                Layer0Utils.addCommentMetadata(graph, "Saved Ontology Definition File " + graph.getRelatedValue2(resource, Layer0.getInstance(graph).HasName, Bindings.STRING));
            }
        });
    }

    @Override
    protected IRunnableContext getOperationRunner(IProgressMonitor monitor) {
        return null;
    }

    @Override
    public boolean isModifiable(Object element) {
        return !errorHappened;
    }

    @Override
    public boolean isReadOnly(Object element) {
        return errorHappened;
    }

}
