/*******************************************************************************
 * Copyright (c) 2012 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.template2d.ui.wizard;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.util.Set;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.SubMonitor;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.simantics.Simantics;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.binding.mutable.Variant;
import org.simantics.databoard.container.DataContainer;
import org.simantics.databoard.container.DataContainers;
import org.simantics.databoard.serialization.SerializationException;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.common.utils.Logger;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.adapter.CopyHandler;
import org.simantics.db.layer0.util.ClipboardUtils;
import org.simantics.db.layer0.util.SimanticsClipboard;
import org.simantics.db.layer0.util.SimanticsClipboard.Representation;
import org.simantics.db.layer0.util.SimanticsClipboardImpl;
import org.simantics.db.layer0.util.SimanticsKeys;
import org.simantics.db.request.Read;
import org.simantics.graph.representation.TransferableGraph1;
import org.simantics.modeling.template2d.DiagramTemplateConstants;
import org.simantics.utils.ui.dialogs.ShowMessage;

/**
 * @author Tuukka Lehtonen
 * @author Teemu Mtsniemi
 */
public class DrawingTemplateExporter implements IRunnableWithProgress {

    ExportPlan exportModel;

    public DrawingTemplateExporter(ExportPlan exportModel) {
        this.exportModel = exportModel;
    }

    @Override
    public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
        SubMonitor progress = SubMonitor.convert(monitor, 50);
        try {
            exportModel(progress.newChild(50, SubMonitor.SUPPRESS_NONE));
        } catch (IOException e) {
            throw new InvocationTargetException(e);
        } catch (DatabaseException e) {
            throw new InvocationTargetException(e);
        } catch (BindingException e) {
            throw new InvocationTargetException(e);
        } finally {
            monitor.done();
        }
    }

    void exportModel(SubMonitor mon) throws IOException, DatabaseException, SerializationException, BindingException{
    	doExport(mon, exportModel.exportLocation, exportModel.model.getResource());
    	mon.setWorkRemaining(0);
    }
    
    public static void doExport(File location, Resource template) throws DatabaseException, IOException {
    	NullProgressMonitor monitor = new NullProgressMonitor();
    	doExport(monitor, location, template);
    }
    
    public static void doExport(IProgressMonitor mon, File location, final Resource template) throws DatabaseException, IOException {
        try {
            // TODO: figure out a way to make the TG go directly into a file
            // instead of having it all in memory at once.
        	
            mon.beginTask("Exporting diagram template...", 100);
            Session session = Simantics.getSession();
            SimanticsClipboard clipboard = session.syncRequest(new Read<SimanticsClipboard>() {
                @Override
                public SimanticsClipboard perform(ReadGraph graph) throws DatabaseException {
                    CopyHandler ch = graph.adapt(template, CopyHandler.class);
                    SimanticsClipboardImpl clipboard = new SimanticsClipboardImpl();
                    ch.copyToClipboard(graph, clipboard);
                    return clipboard;
                }
            });
            for (Set<Representation> object : clipboard.getContents()) {
                TransferableGraph1 tg = ClipboardUtils.accept(object, SimanticsKeys.KEY_TRANSFERABLE_GRAPH);
                mon.worked(95);

                mon.setTaskName("Writing transferable graph...");
                DataContainers.writeFile(location, new DataContainer(
                        DiagramTemplateConstants.DRAWING_TEMPLATE_FORMAT, DiagramTemplateConstants.DRAWING_TEMPLATE_CURRENT_VERSION,
                        new Variant(TransferableGraph1.BINDING, tg)));

                mon.worked(5);
            }

        } catch (DatabaseException e) {
            e.printStackTrace();
            Logger.defaultLogError(e);
            mon.setCanceled(true);
            ShowMessage.showError("Export failed.", "Internal application error in export. See log for details.");
        }
    }

}
