/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.mapping;

import java.util.Map;
import java.util.function.BiFunction;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Statement;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.common.utils.OrderedSetUtils;
import org.simantics.db.exception.CancelTransactionException;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.request.PossibleModel;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.diagram.stubs.G2DResource;
import org.simantics.diagram.synchronization.ISynchronizationContext;
import org.simantics.diagram.synchronization.StatementEvaluation;
import org.simantics.diagram.synchronization.graph.BasicResources;
import org.simantics.diagram.synchronization.graph.CopyAdvisorUtil;
import org.simantics.diagram.synchronization.graph.GraphCopyAdvisor;
import org.simantics.diagram.synchronization.graph.GraphSynchronizationHints;
import org.simantics.layer0.Layer0;

import gnu.trove.map.hash.THashMap;

/**
 * @author Tuukka Lehtonen
 */
public class ElementCopyAdvisor extends GraphCopyAdvisor {

    G2DResource G2D;
    DiagramResource DIA;

    /**
     * This is necessary to have DIA.Flag type copied over to the copied flag.
     * Diagram mapping will have problems and potentially break the
     * configuration if the type is not the same as in the source.
     */
    BiFunction<ReadGraph, Statement, StatementEvaluation> statementAdvisor =
            new BiFunction<ReadGraph, Statement, StatementEvaluation>() {
        @Override
        public StatementEvaluation apply(ReadGraph graph, Statement stm) {
            if (DIA.HasFlagType.equals(stm.getPredicate()))
                return StatementEvaluation.INCLUDE;
            if (G2D.HasFontStyle.equals(stm.getPredicate()))
                return StatementEvaluation.INCLUDE;
            return StatementEvaluation.USE_DEFAULT;
        }
    };

    @Override
    public Evaluation canCopy(ISynchronizationContext context, WriteGraph graph, Resource source, Resource sourcecontainer, Resource targetContainer) throws DatabaseException {
        BasicResources br = context.get(GraphSynchronizationHints.BASIC_RESOURCES);
        return graph.isInstanceOf(source, br.DIA.Element) ? Evaluation.SUPPORTED : Evaluation.NOT_SUPPORTED;
    }

    @Override
    public Object copy(ISynchronizationContext context, WriteGraph graph, Resource source, Resource sourceContainer, Resource targetContainer) throws DatabaseException {
        return copy(context, graph, source, sourceContainer, targetContainer, new THashMap<Object, Object>());
    }

    @Override
    public Object copy(ISynchronizationContext context, WriteGraph graph, Resource source, Resource sourceContainer,
            Resource targetContainer, Map<Object, Object> map) throws DatabaseException {
        Resource model = graph.syncRequest(new PossibleModel(targetContainer));
        //if (model == null)
        //    throw new IllegalArgumentException("no model found for copy target container " + NameUtils.getSafeName(graph, targetContainer, true));

        Layer0 L0 = Layer0.getInstance(graph);
        G2D = G2DResource.getInstance(graph);
        DIA = DiagramResource.getInstance(graph);

        Resource copy = (model != null) ?
                CopyAdvisorUtil.copy3(graph, source, model, statementAdvisor, map) :
                    CopyAdvisorUtil.copy2(graph, source, statementAdvisor, map);

        if (graph.hasStatement(sourceContainer, L0.ConsistsOf, source))
            graph.claim(targetContainer, L0.ConsistsOf, copy);

        return copy;
    }

    @Override
    public Object cut(ISynchronizationContext context, WriteGraph graph, Resource source, Resource sourceContainer,
            Resource targetContainer) throws DatabaseException {

    	Layer0 L0 = Layer0.getInstance(graph);
    	DiagramResource DIA = DiagramResource.getInstance(graph);

        if (sourceContainer.equals(targetContainer))
            return null;

        // disconnect from source diagram
        if (!OrderedSetUtils.remove(graph, sourceContainer, source))
            // Unlink failed for some reason.
            throw new CancelTransactionException("Failed to unlink element "
                    + NameUtils.getSafeName(graph, source) + " from source diagram "
                    + NameUtils.getSafeName(graph, sourceContainer));

        // connect to target diagram
        boolean prepend = graph.isInstanceOf(source, DIA.Connection);
        boolean add = true;
        if (prepend)
            add = OrderedSetUtils.addFirst(graph, targetContainer, source);
        else
            add = OrderedSetUtils.add(graph, targetContainer, source);
        if (!add)
            // Link failed for some reason.
            throw new CancelTransactionException("Failed to link element "
                    + NameUtils.getSafeName(graph, source) + " to target diagram "
                    + NameUtils.getSafeName(graph, targetContainer));

        // This handles e.g. connections, which are not part of container
        if(graph.hasStatement(sourceContainer, L0.ConsistsOf, source)) {
            graph.deny(sourceContainer, L0.ConsistsOf, source);
            graph.claim(targetContainer, L0.ConsistsOf, source);
        }

        return source;
    }

}
