/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.adapters;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.ObjectsWithType;
import org.simantics.db.common.request.PossibleIndexRoot;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.adapter.SubgraphExtent.ExtentStatus;
import org.simantics.db.layer0.adapter.impl.DefaultCopyHandler;
import org.simantics.db.layer0.util.TransferableGraphConfiguration2;
import org.simantics.db.layer0.util.TransferableGraphConfiguration2.RootSpec;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ModelingResources;
import org.simantics.modeling.ModelingUtils.CompositeInfo;
import org.simantics.modeling.ModelingUtils.DiagramComponentInfo;
import org.simantics.structural.stubs.StructuralResource2;

/**
 * @author Tuukka Lehtonen
 */
public class CompositeCopyHandler extends DefaultCopyHandler {

    public CompositeCopyHandler(Resource composite) {
    	super(composite);
    }

    public CompositeCopyHandler(Collection<Resource> composites) {
        super(composites);
    }

    @Override
    protected TransferableGraphConfiguration2 createConfiguration(ReadGraph graph, boolean cut) throws DatabaseException {

        Layer0 L0 = Layer0.getInstance(graph);
        StructuralResource2 SR = StructuralResource2.getInstance(graph);
        DiagramResource DIA = DiagramResource.getInstance(graph);
        ModelingResources MOD = ModelingResources.getInstance(graph);

        Set<Resource> resourceSet = (resources instanceof Set)
                ? (Set<Resource>) resources : new HashSet<>(resources);
        Set<Resource> exclusions = new HashSet<>();
        Set<Resource> externals = new HashSet<>();
        List<RootSpec> roots = new ArrayList<>();

        for(Resource resource : resources) {
            // Process all connection joins.
            // This is the only way to access all of them.
            for (Resource diagram : graph.getObjects(resource, MOD.CompositeToDiagram)) {
                for (Resource flag : graph.syncRequest(new ObjectsWithType(diagram, L0.ConsistsOf, DIA.Flag))) {
                    for (Resource join : graph.getObjects(flag, DIA.FlagIsJoinedBy)) {
                        // Joins with external references are omitted
                        for (Resource comp : graph.getObjects(join, SR.JoinsComposite)) {
                            if (!resourceSet.contains(comp))
                                exclusions.add(join);
                        }
                        // This code excludes joins with flags to external
                        // diagrams that are not connected (have no
                        // configuration for the flag)
                        for (Resource flag2 : graph.getObjects(join, DIA.JoinsFlag)) {
                            Resource diagram2 = graph.getPossibleObject(flag2, L0.PartOf);
                            if (diagram2 != null) {
                                Resource comp = graph.getPossibleObject(diagram2, MOD.DiagramToComposite);
                                if (!resourceSet.contains(comp))
                                    exclusions.add(join); 
                            }
                        }
                    }
                }

                // Check all diagram monitor elements.
                // Any components referenced that are external to the exported diagrams must be excluded from the export.
                // This will leave the monitors without a monitored component but export and import will work anyway.
                for (Resource ref : graph.syncRequest(new ObjectsWithType(diagram, L0.ConsistsOf, DIA.Monitor))) {
                    for (Resource monitoredComponent : graph.getObjects(ref, DIA.HasMonitorComponent)) {
                        Resource monitoredComponentComposite = graph.getPossibleObject(monitoredComponent, L0.PartOf);
                        if (monitoredComponentComposite != null && !resourceSet.contains(monitoredComponentComposite)) {
                            exclusions.add(monitoredComponent);
                        }
                    }
                }
            }

            // Leave any diagram monitors out of the export that are contained
            // by diagrams that are not part of this export, when the monitored
            // components happen to be a part of the exported composites.
            for (Resource ref : graph.syncRequest(new ObjectsWithType(resource, L0.ConsistsOf, SR.Component))) {
                for (Resource monitor : graph.getObjects(ref, DIA.HasMonitorComponent_Inverse)) {
                    Resource monitorDiagram = graph.getPossibleObject(monitor, L0.PartOf);
                    if (monitorDiagram != null) {
                        Resource monitorComposite = graph.getPossibleObject(monitorDiagram, MOD.DiagramToComposite);
                        if (monitorComposite != null && !resourceSet.contains(monitorComposite))
                            exclusions.add(monitor);
                    }
                }
            }

            // Include resource as root
//            GUID rootId = graph.getRelatedValue(resource, L0.identifier, GUID.BINDING);
//            String rootName = graph.getRelatedValue(resource, L0.HasName, Bindings.STRING);
//            String escapedRootName = URIStringUtils.escape(rootName);
//            String escapedPath = ModelingUtils.getDiagramCompositePath(graph, resource);
            CompositeInfo info = CompositeInfo.fromResource(graph, resource);
            roots.add(new RootSpec(resource, info.getTGName(), true));
        	Resource id = graph.getPossibleObject(resource, L0.identifier);
        	if(id != null) exclusions.add(id);
            // Include components as roots
            for(Resource child : graph.sync(new ObjectsWithType(resource, L0.ConsistsOf, SR.Component))) {
            	DiagramComponentInfo cinfo = DiagramComponentInfo.fromResource(graph, info, child);
//            	GUID childId = graph.getRelatedValue(resource, L0.identifier, GUID.BINDING);
//                String childName = graph.getRelatedValue(child, L0.HasName, Bindings.STRING);
            	id = graph.getPossibleObject(child, L0.identifier);
            	if(id != null) exclusions.add(id);
                roots.add(new RootSpec(child, cinfo.getTGName(info), true));
            }
        }

        Resource model = graph.syncRequest(new PossibleIndexRoot(resources.iterator().next()));
        if(model == null) throw new DatabaseException("Composite is not part of any index root");
        roots.add(new RootSpec(model, "%model", false));

        TransferableGraphConfiguration2 config = TransferableGraphConfiguration2.createWithNames2(graph, roots, exclusions, true, false);
        for (Resource external : externals)
            config.preStatus.put(external, ExtentStatus.EXTERNAL);
        return config;
    }

}
