/*******************************************************************************
 * Copyright (c) 2014, 2015 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling;

import gnu.trove.set.hash.THashSet;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Set;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.simantics.Simantics;
import org.simantics.databoard.Bindings;
import org.simantics.databoard.util.URIStringUtils;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.NamedResource;
import org.simantics.db.common.request.ObjectsWithType;
import org.simantics.db.common.request.PossibleIndexRoot;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.common.utils.VersionMap;
import org.simantics.db.common.utils.Versions;
import org.simantics.db.exception.CancelTransactionException;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.adapter.Instances;
import org.simantics.db.layer0.request.ActiveModels;
import org.simantics.db.layer0.util.Layer0Utils;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.MigrateModel.MigrationOperation;
import org.simantics.modeling.migration.UserComponentPostMigrationAction;
import org.simantics.simulation.ontology.SimulationResource;
import org.simantics.structural.stubs.StructuralResource2;
import org.simantics.utils.datastructures.MapList;
import org.simantics.utils.datastructures.Pair;
import org.simantics.utils.datastructures.Triple;
import org.simantics.utils.strings.AlphanumComparator;

/**
 * @author Antti Villberg
 * @author Tuukka Lehtonen
 */
public class UserComponentMigration {

    public static void migrateUserComponents(WriteGraph graph, Resource source, Resource target, Collection<Resource> components) throws DatabaseException {
        MigrateModel model = getComponentTypeModel(graph, source, target, null);
        if (model.instances.isEmpty())
            return;
        Triple<String, NamedResource, Collection<MigrationOperation>> instances = model.instances.get(0);
        ArrayList<MigrationOperation> result = new ArrayList<>();
        
        for (MigrationOperation instance : instances.third) {
            if (components.contains(instance.instanceToMigrate.getResource())) {
               result.add(instance); 
            }
        }
        if (!result.isEmpty())
            doMigration(new NullProgressMonitor(), graph, result);
    }

    public static String doMigration(IProgressMonitor monitor, WriteGraph graph, final ArrayList<MigrationOperation> operations) throws DatabaseException {
        graph.markUndoPoint();
        StringBuilder b = new StringBuilder();
        int success = 0;
        int problem = 0;
        int no = 1;
        int count = operations.size();
        monitor.setTaskName("Migrating " + count + " User Components");
        for(MigrationOperation op : operations) {
            if (monitor.isCanceled())
                throw new CancelTransactionException();
            monitor.subTask("(" + (no++) + "/" + count + ") " + op.toString());
            String problems = op.perform(graph);
            if(problems != null) {
                b.insert(0, problems);
                b.insert(0, op.getDescription(graph) + "\n");
                problem++;
            } else {
                b.append(op.getDescription(graph) + "\n");
                b.append(" success\n");
                success++;
            }
        }
        int total = success + problem;
        b.insert(0, "---------------------\n");
        b.insert(0, "Details:\n");
        b.insert(0, "\n");
        b.insert(0, "  OK: " + success + "\n"); 
        b.insert(0, "  Failure: " + problem + "\n"); 
        b.insert(0, "---------------------\n");
        b.insert(0, "Performed migration for " + total + " instances:\n");

        Layer0Utils.addCommentMetadata(graph, "Migrated " + total + " instances");
        return b.toString();
    }

    public static void doPostMigration(IProgressMonitor monitor, WriteGraph graph, ArrayList<MigrationOperation> result) throws DatabaseException {
        THashSet<Resource> roots = new THashSet<>();
        for(MigrationOperation op : result) {
            Resource root = graph.syncRequest(new PossibleIndexRoot(op.instanceToMigrate.getResource()));
            if(root != null)
                roots.add(root);
        }
        if (monitor.isCanceled())
            throw new CancelTransactionException();
        for(Resource root : roots) {
            UserComponentPostMigrationAction action = graph.getPossibleAdapter(root, UserComponentPostMigrationAction.class);
            if(action != null)
                action.perform(monitor, graph);
        }
    }

    public static MigrateModel getComponentTypeModel(ReadGraph graph, Resource source, Resource target, Resource symbol) throws DatabaseException {
        MigrateModel model = newMigrateModel(graph);

        MapList<NamedResource, MigrationOperation> list = new MapList<>(); 

        Layer0 L0 = Layer0.getInstance(graph);
        DiagramResource DIA = DiagramResource.getInstance(graph);
        Instances query = graph.adapt(source, Instances.class);
        Set<Resource> instances = new THashSet<>();
        for(NamedResource nr : getLocations(graph, Simantics.getProjectResource())) {
            Collection<Resource> found = query.find(graph, nr.getResource());
            instances.addAll(found);
        }

        model.instanceCount = instances.size();
        for(Resource instance : instances) {

            String uri = graph.getPossibleURI(instance);

            Resource element = ModelingUtils.getPossibleElement(graph, instance);
            if(element == null) {
                MigrationOperation op = new MigrationOperation(new NamedResource(uri, instance), new NamedResource("", target), null);
                addInstance(list, graph, instance, op);
                continue;
            }

            Resource instanceSymbol = graph.getPossibleType(element, DIA.Element);
            if(instanceSymbol == null) continue;

            if(symbol != null) {
                if(!symbol.equals(instanceSymbol)) {
                    MigrationOperation op = new MigrationOperation(new NamedResource(uri, instance), new NamedResource("", target), new NamedResource("", symbol));
                    addInstance(list, graph, instance, op);
                }
            } else {
                String instanceSymbolName = graph.getRelatedValue(instanceSymbol, L0.HasName, Bindings.STRING);
                Resource targetSymbol = Layer0Utils.getPossibleChild(graph, target, DIA.ElementClass, instanceSymbolName); 
                if(targetSymbol != null && !targetSymbol.equals(instanceSymbol)) {
                    MigrationOperation op = new MigrationOperation(new NamedResource(uri, instance), new NamedResource("", target), new NamedResource("", targetSymbol));
                    addInstance(list, graph, instance, op);
                }
            }

        }

        sortInstances(model, list);

        return model;
    }

    public static MigrateModel getSharedOntologyModel(ReadGraph graph, Resource sourceOntology, Resource targetOntology) throws DatabaseException {
        MigrateModel model = newMigrateModel(graph);

        MapList<NamedResource, MigrationOperation> list = new MapList<>(); 

        Layer0 L0 = Layer0.getInstance(graph);
        DiagramResource DIA = DiagramResource.getInstance(graph);
        StructuralResource2 STR = StructuralResource2.getInstance(graph);
        Instances query = graph.adapt(STR.ComponentType, Instances.class);

        Set<Resource> types = new THashSet<>();
        for(Resource type : query.find(graph, sourceOntology)) {
            // TODO: haxx
            if(graph.isInheritedFrom(type, DIA.Element)) continue;
            Resource root = graph.syncRequest(new PossibleIndexRoot(type));
            if(sourceOntology.equals(root)) types.add(type);
        }

        Set<Resource> instances = new THashSet<>();
        Collection<NamedResource> locations = getLocations(graph, Simantics.getProjectResource()); 

        for(Resource type : types) {
            Instances query2 = graph.adapt(type, Instances.class);
            for(NamedResource nr : locations) {
                Collection<Resource> found = query2.find(graph, nr.getResource());
                instances.addAll(found);
            }
        }

        model.instanceCount = instances.size();
        for(Resource instance : instances) {
            Resource type = graph.getPossibleType(instance, STR.Component);
            String uri = graph.getPossibleURI(instance);
            if (type == null || uri == null) {
                System.err.println("CANNOT MIGRATE INSTANCE DUE TO TYPING PROBLEM: " + NameUtils.getURIOrSafeNameInternal(graph, instance));
                continue;
            }
            NamedResource best = matchBest(graph, type, targetOntology);
            if(best != null) {

                Resource element = ModelingUtils.getPossibleElement(graph, instance);
                if(element == null) {
                    MigrationOperation op = new MigrationOperation(new NamedResource(uri, instance), best, null);
                    addInstance(list, graph, instance, op);
                    continue;
                }

                Resource instanceSymbol = graph.getPossibleType(element, DIA.Element);
                if(instanceSymbol == null) continue;

                String instanceSymbolName = graph.getRelatedValue(instanceSymbol, L0.HasName, Bindings.STRING);
                Resource targetSymbol = Layer0Utils.getPossibleChild(graph, best.getResource(), DIA.ElementClass, instanceSymbolName); 
                if(targetSymbol != null && !targetSymbol.equals(instanceSymbol)) {
                    MigrationOperation op = new MigrationOperation(new NamedResource(uri, instance), best, new NamedResource("", targetSymbol));
                    addInstance(list, graph, instance, op);
                }

            }
        }

        sortInstances(model, list);

        return model;
    }

    private static Collection<NamedResource> getLocations(ReadGraph graph, final Resource project) throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(graph);
        SimulationResource SIMU = SimulationResource.getInstance(graph);
        Collection<NamedResource> libraries = new ArrayList<>();
        for (Resource r : graph.syncRequest(new ObjectsWithType(project, L0.ConsistsOf, SIMU.Model))) {
            String name = Versions.getStandardNameString(graph, r);
            libraries.add(new NamedResource(name, r));
        }
        Collection<Resource> ontologies = Simantics.applySCL("Simantics/SharedOntologies", "traverseSharedOntologies", graph, graph.getRootLibrary());
        for (Resource r : ontologies) {
            String name = Versions.getStandardNameString(graph, r);
            libraries.add(new NamedResource(name, r));
        }
        return libraries;
    }

    private static final Comparator<NamedResource> NAMED_RESOURCE_COMPARATOR = new Comparator<NamedResource>() {
        @Override
        public int compare(NamedResource o1, NamedResource o2) {
            return AlphanumComparator.CASE_INSENSITIVE_COMPARATOR.compare(o1.getName(), o2.getName());
        }
    };

    private static void sortInstances(MigrateModel model, MapList<NamedResource, MigrationOperation> list) {
        ArrayList<NamedResource> keys = new ArrayList<>(list.getKeys());
        Collections.sort(keys, NAMED_RESOURCE_COMPARATOR);
        for(NamedResource key : keys) {
            Collection<MigrationOperation> ops = list.getValuesSnapshot(key);
            String[] parts = key.getName().split("#");
            //System.out.println("Parts: " + Arrays.toString(parts));
            model.instances.add(Triple.make(parts[0], new NamedResource(URIStringUtils.unescape(parts[1]), key.getResource()), ops));
        }
    }

    private static Pair<String,Integer> addInstance(MapList<NamedResource, MigrationOperation> list, ReadGraph graph, Resource instance, MigrationOperation op) throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(graph);
        if(graph.isInstanceOf(instance, L0.IndexRoot)) return Pair.make("", 0);
        instance = graph.getPossibleObject(instance, L0.PartOf);
        if(instance == null) return Pair.make("", 0);
        String name = Versions.getStandardNameString(graph, instance);
        String escapedName = URIStringUtils.escape(name);
        Pair<String,Integer> parent = addInstance(list, graph, instance, op);
        StringBuilder code = new StringBuilder(parent.first).append('/').append(escapedName).append('#');
        for(int i=0;i<parent.second;i++) code.append(' ');
        code.append(escapedName);
        list.add(new NamedResource(code.toString(), instance), op);
        return Pair.make(parent.first + "/" + escapedName, parent.second + 4);
    }

    private static NamedResource matchBest(ReadGraph graph, Resource type, Resource newOntology) throws DatabaseException {
        VersionMap versions = Versions.match(graph, type, newOntology);
        return versions.getNewest(graph, Versions.getBaseName(graph, type));
    }

    private static MigrateModel newMigrateModel(ReadGraph graph) throws DatabaseException {
        MigrateModel model = new MigrateModel();
        model.activeModels.addAll( graph.syncRequest(new ActiveModels(Simantics.getProjectResource())) );
        return model;
    }

}
