package org.simantics.graph.compiler.internal.translation;

import gnu.trove.map.hash.TObjectIntHashMap;
import gnu.trove.set.hash.TIntHashSet;

import java.util.ArrayList;
import java.util.Collection;

import org.antlr.runtime.tree.Tree;
import org.simantics.graph.compiler.SourceInfo.DefinitionPosition;
import org.simantics.graph.compiler.internal.parsing.GraphParser;
import org.simantics.graph.compiler.internal.store.LocationStore;
import org.simantics.graph.compiler.internal.store.PreValueStore;
import org.simantics.graph.compiler.internal.templates.TemplateDefinition;
import org.simantics.graph.compiler.internal.templates.TemplateDefinitionStore;
import org.simantics.graph.compiler.internal.templates.TemplateInstanceStore;
import org.simantics.graph.compiler.internal.values.TreeValue;
import org.simantics.graph.query.Paths;
import org.simantics.graph.query.UriUtils;
import org.simantics.graph.store.GraphStore;
import org.simantics.ltk.ISource;
import org.simantics.ltk.Problem;
import org.simantics.ltk.antlr.ANTLRUtils;

public class GraphTranslator extends GraphStore {

	ISource source;
	Collection<Problem> problems;
	
	TObjectIntHashMap<String> variables = new TObjectIntHashMap<String>();
	ArrayList<DefinitionPosition> definitionPositions = new ArrayList<DefinitionPosition>();
	TIntHashSet definedResources = new TIntHashSet();
	
	LocationStore locations;
	TemplateInstanceStore templateInstances;
	TemplateDefinitionStore templateDefinitions;
	PreValueStore preValues;
	
	Paths paths;
	int layer0;
	
	public GraphTranslator(Paths paths, Collection<Problem> problems, GraphStore store) {
		super(store);		
		this.problems = problems;
		this.paths = paths;
		
		locations = getStore(LocationStore.class);
		if(locations == null) {
			locations = new LocationStore();
			addStore(LocationStore.class, locations);
		}
		
		templateInstances = getStore(TemplateInstanceStore.class);
		if(templateInstances == null) {
			templateInstances = new TemplateInstanceStore();
			addStore(TemplateInstanceStore.class, templateInstances);
		}
		
		templateDefinitions = getStore(TemplateDefinitionStore.class);
		if(templateDefinitions == null) {
			templateDefinitions = new TemplateDefinitionStore();
			addStore(TemplateDefinitionStore.class, templateDefinitions);
		}
		
		preValues = getStore(PreValueStore.class);
		if(preValues == null) {
			preValues = new PreValueStore();
			addStore(PreValueStore.class, preValues);
		}
			
		layer0 = identities.createPathToId(paths.Layer0);		
	}
	
	public void setSource(ISource source) {
		this.source = source;
	}
	
	public void translateGraph(Tree tree) {		
		int count = tree.getChildCount();
		for(int i=0;i<count;++i)
			translateResourceDefinition(tree.getChild(i));
	}

	protected void locate(int resource, Tree tree) {
		locations.add(resource, ANTLRUtils.location(source, tree));
	}
	
	public int translateResourceDefinition(Tree tree) {
		Tree subjectTree = tree.getChild(0);
		int subject = translateResource(subjectTree);
		
		int count = tree.getChildCount();
		for(int i=1;i<count;++i) {
			Tree child = tree.getChild(i);
			switch(child.getType()) {
			case GraphParser.PROPERTY:
				translateProperty(subject, subjectTree, child);
				break;
			case GraphParser.TEMPLATE_INSTANCE:
				translateTemplateInstance(subject, child);
				break;
			case GraphParser.TEMPLATE_DEFINITION:
				translateTemplateDefinition(subject, child);
				break;
			}
		}
		return subject;
	}

	private void translateProperty(int subject, Tree subjectTree, Tree tree) {
		int predicate = translateResource(tree.getChild(0));
		if(predicate == instanceOfId || predicate == inheritsId || predicate == subrelationOfId)
			if(definedResources.add(subject)) {				
		        definitionPositions.add(new DefinitionPosition(subject,
		        		ANTLRUtils.getStartLine(subjectTree) + source.startLine(),
		        		ANTLRUtils.getStartIndex(subjectTree) + source.startPos(),
		        		ANTLRUtils.getStopIndex(subjectTree) + source.startPos()
		        ));
			}
		int count = tree.getChildCount();
		for(int i=1;i<count;++i) {
			int object = translateResourceDefinition(tree.getChild(i));
			statements.add(subject, predicate, object);
		}
	}	
	 
	private void translateTemplateInstance(int subject, Tree tree) {
		int parameterCount = tree.getChildCount();
		int[] templateInstance = new int[parameterCount+1];
		templateInstance[0] = translateResource(tree.getChild(0));
		templateInstance[1] = subject;
		for(int i=1;i<parameterCount;++i) {
			Tree child = tree.getChild(i);
			if(child.getType() == GraphParser.RESOURCE)
				templateInstance[i+1] = translateResourceDefinition(child);
			else
				templateInstance[i+1] = translateResource(child);
		}
		templateInstances.add(templateInstance);
	}	
	
	private void translateTemplateDefinition(int subject, Tree tree) {
		ArrayList<String> parameters = new ArrayList<String>(); 

		int i=0;
		while(i < tree.getChildCount() && 
				tree.getChild(i).getType() == GraphParser.VARIABLE) {
			parameters.add(tree.getChild(i).getChild(0).getText());
			++i;
		}

		GraphStore store = new GraphStore();
		ChildGraphTranslator translator = new ChildGraphTranslator(this, store);
		translator.setSource(source);
		while(i < tree.getChildCount()) {
			translator.translateResourceDefinition(tree.getChild(i));
			++i;
		}	
		
		templateDefinitions.add(subject, 
				new TemplateDefinition(parameters.toArray(new String[parameters.size()]), 
						translator.correspondences.toArray(), 
						store));
	}

	public static void printTree(Tree tree) {
		printTree(0, tree);
	}
	
	int equalsId = 0;
	int inheritsId = 0;
	int subrelationOfId = 0;
	int instanceOfId = 0;
	int hasDomainId = 0;
	int hasRangeId = 0;
	int domainOfId = 0;
	int requiresValueTypeId = 0;
	int DatatypeId = 0;
	
	private int translateResource(Tree tree) {
		switch(tree.getType()) {
		case GraphParser.EMBEDDED_VALUE: {
			int id = identities.newResource();
			preValues.setValue(id, new TreeValue(source, tree.getChild(0)));
			locate(id, tree.getChild(0));
			return id;
		}
		case GraphParser.EMBEDDED_TYPE: {
			int DataType = identities.newResource();
			values.setValue(DataType, new DataTypeTranslator(source, problems).translate(tree.getChild(0)));
			if(instanceOfId == 0)
				instanceOfId = identities.getChild(layer0, "InstanceOf");
			if(DatatypeId == 0)
				DatatypeId = identities.getChild(layer0, "DataType");
			statements.add(DataType, instanceOfId, DatatypeId);
			locate(DataType, tree.getChild(0));
			return DataType;
		}
		case GraphParser.BLANK: {
			int id = identities.newResource();
			locate(id, tree);
			return id;
		}
		case GraphParser.ID:
			return getVariable(tree, tree.getText());
		case GraphParser.VARIABLE: {
			int id = identities.getRoot(tree.getChild(0).getText());
			locate(id, tree);
			return id;
		}
		case GraphParser.URI: {
			String uri = tree.getText();
			uri = uri.substring(1, uri.length()-1);
			int id = identities.createPathToId(UriUtils.uriToPath(uri));
			locate(id, tree);
			return id;
		}
		case GraphParser.REF: {
			int parent = translateResource(tree.getChild(0));
			Tree nameNode = tree.getChild(1);
			String name = nameNode.getText();
			if(nameNode.getType() == GraphParser.STRING) {
				name = name.substring(1, name.length()-1);
			}
			int id = identities.getChild(parent, name);
			locate(id, tree);
			return id;
		}
		case GraphParser.EQUALS:
			if(equalsId == 0)
				equalsId = identities.getChild(layer0, "Equals");
			return equalsId;
		case GraphParser.INHERITS:
			if(inheritsId == 0)
				inheritsId = identities.getChild(layer0, "Inherits");
			return inheritsId;
		case GraphParser.SUBRELATION_OF:
			if(subrelationOfId == 0)
				subrelationOfId = identities.getChild(layer0, "SubrelationOf");
			return subrelationOfId;
		case GraphParser.INSTANCE_OF:
			if(instanceOfId == 0)
				instanceOfId = identities.getChild(layer0, "InstanceOf");
			return instanceOfId;		
		case GraphParser.HAS_DOMAIN:
            if(hasDomainId == 0)
                hasDomainId = identities.getChild(layer0, "HasDomain");
            return hasDomainId;			
		case GraphParser.HAS_RANGE:
            if(hasRangeId == 0)
                hasRangeId = identities.getChild(layer0, "HasRange");
            return hasRangeId;
		case GraphParser.DOMAIN_OF:
            if(domainOfId == 0)
                domainOfId = identities.getChild(layer0, "DomainOf");
            return domainOfId;     
        case GraphParser.REQUIRES_VALUE_TYPE:
            if(requiresValueTypeId == 0)
                requiresValueTypeId = identities.getChild(layer0, "RequiresValueType");
            return requiresValueTypeId;            
		default: 
			// TODO
			System.out.println("-- " + GraphParser.tokenNames[tree.getType()]);
			printTree(tree);
			return 0;
		}		
	}

	public int getVariable(Tree place, String text) {
		if(variables.containsKey(text))
			return variables.get(text);
		else {
			int id = identities.newResource();
			variables.put(text, id);
			locate(id, place);
			return id;
		}
	}
	
	public TObjectIntHashMap<String> getVariables() {
		return variables;
	}
	
	public ArrayList<DefinitionPosition> getDefinitionPositions() {
		return definitionPositions;
	}

	public static void printTree(int indent, Tree tree) {
		for(int i=0;i<indent;++i)
			System.out.print("  ");
		System.out.println(tree.getText());
		for(int i=0;i<tree.getChildCount();++i)
			printTree(indent+1, tree.getChild(i));
	}	
}
