package org.simantics.graph.compiler.internal.translation;

import gnu.trove.map.hash.THashMap;

import java.util.Collection;

import org.antlr.runtime.tree.Tree;
import org.simantics.databoard.Datatypes;
import org.simantics.databoard.type.ArrayType;
import org.simantics.databoard.type.ByteType;
import org.simantics.databoard.type.Component;
import org.simantics.databoard.type.Datatype;
import org.simantics.databoard.type.DoubleType;
import org.simantics.databoard.type.FloatType;
import org.simantics.databoard.type.IntegerType;
import org.simantics.databoard.type.LongType;
import org.simantics.databoard.type.MapType;
import org.simantics.databoard.type.OptionalType;
import org.simantics.databoard.type.RecordType;
import org.simantics.databoard.type.StringType;
import org.simantics.databoard.type.UnionType;
import org.simantics.graph.compiler.internal.parsing.GraphParser;
import org.simantics.ltk.ISource;
import org.simantics.ltk.Problem;
import org.simantics.ltk.antlr.ANTLRUtils;

public class DataTypeTranslator {

	ISource source;
	Collection<Problem> problems;
		
	public DataTypeTranslator(ISource source, Collection<Problem> problems) {
		this.source = source;
		this.problems = problems;
	}

	public Datatype translate(Tree tree) {		
		switch(tree.getType()) {
		case GraphParser.RECORD_TYPE:
			return translateRecordType(tree);
		case GraphParser.UNION_TYPE:
			return translateUnionType(tree);
		case GraphParser.TUPLE_TYPE:
			return translateTupleType(tree);
		case GraphParser.ARRAY_TYPE:
			return translateArrayType(tree);
		case GraphParser.TYPE_REFERENCE:
			return translateTypeReference(tree);
		default:
			printTree(0, tree);
			throw new IllegalArgumentException("The argument is not a data type AST");
		}
	}

	private Datatype translateArrayType(Tree tree) {
		return new ArrayType(translate(tree.getChild(0)));
	}

	private Datatype translateTupleType(Tree tree) {
		int count = tree.getChildCount(); 
		if(count == 1)
			return translate(tree.getChild(0));
		Component[] components = new Component[count];
		for(int i=0;i<count;++i) {
			components[i] = new Component(
					Integer.toString(i),
					translate(tree.getChild(i))
				);
		}
		return new RecordType(false, components);		
	}
	
	private static THashMap<String,String> getAnnotations(Tree tree) {
		THashMap<String,String> result = new THashMap<String,String>(); 
		for(int i=1;i<tree.getChildCount();++i) {
			Tree child = tree.getChild(i);
			if(child.getType() == GraphParser.TYPE_ANNOTATION) {
				result.put(
						child.getChild(0).getText(),
						convertValue(child.getChild(1))
				);
			}
		}
		return result;
	}
	
	private static String convertValue(Tree value) {
		//System.out.println(GraphParser.tokenNames[value.getType()]);
		switch(value.getType()) {
		case GraphParser.STRING: {
			String str = value.getText();
			return str.substring(1, str.length()-1);
		}
		case GraphParser.INT:
		case GraphParser.FLOAT:
			return value.getText();
		case GraphParser.RANGE:
			return value.getChild(0).getText() + value.getChild(1).getText() + value.getChild(2).getText();
		default:
			throw new IllegalArgumentException();
		}
	}

	private Datatype translateTypeReference(Tree tree) {
		String name = tree.getChild(0).getText();
		if(name.equals("Boolean"))
			return Datatypes.BOOLEAN;
		else if(name.equals("Byte")) {
			THashMap<String,String> annotations = getAnnotations(tree);
			return new ByteType(
					annotations.get("unit"),
					annotations.get("range")
			);
		}
		else if(name.equals("Integer")) {
			THashMap<String,String> annotations = getAnnotations(tree);
			return new IntegerType(
					annotations.get("unit"),
					annotations.get("range")
			);
		}
		else if(name.equals("Float")) {
			THashMap<String,String> annotations = getAnnotations(tree);
			return new FloatType(
					annotations.get("unit"),
					annotations.get("range")
			);
		}
		else if(name.equals("Double")) {
			THashMap<String,String> annotations = getAnnotations(tree);
			return new DoubleType(
					annotations.get("unit"),
					annotations.get("range")
			);
		}
		else if(name.equals("Long")) {
			THashMap<String,String> annotations = getAnnotations(tree);
			return new LongType(
					annotations.get("unit"),
					annotations.get("range")
			);
		}
		else if(name.equals("String")) {
			THashMap<String,String> annotations = getAnnotations(tree);
			return new StringType(
					annotations.get("pattern"),
					annotations.get("mimeType"),
					annotations.get("length")
			);
		}
		else if(name.equals("Variant"))
			return Datatypes.VARIANT;
		else if(name.equals("Optional"))
			return new OptionalType(translate(tree.getChild(1)));
		else if(name.equals("Map"))
			return new MapType(translate(tree.getChild(1)), translate(tree.getChild(2)));
		else if(name.equals("DataType"))
			return Datatypes.getDatatypeUnchecked(Datatype.class);
		else { 
			error(tree, "Invalid data type " + name + ".");
			return null;
		}
	}

	private Datatype translateUnionType(Tree tree) {
		Component[] components = new Component[tree.getChildCount()];
		for(int i=0;i<components.length;++i) {
			Tree c = tree.getChild(i);
			components[i] = new Component(
					c.getChild(0).getText(), 
					c.getChildCount() > 1 ? translate(c.getChild(1)) : RecordType.VOID_TYPE
				);
		}
		return new UnionType(components);
	}

	private Datatype translateRecordType(Tree tree) {
		Component[] components = new Component[tree.getChildCount()];
		for(int i=0;i<components.length;++i) {
			Tree c = tree.getChild(i);
			components[i] = new Component(
					c.getChild(0).getText(), 
					translate(c.getChild(1))
				);
		}
		return new RecordType(false, components);
	}
	
	public static void printTree(int indent, Tree tree) {
		for(int i=0;i<indent;++i)
			System.out.print("  ");
		System.out.println(tree.getText());
		for(int i=0;i<tree.getChildCount();++i)
			printTree(indent+1, tree.getChild(i));
	}
	
	private void error(Tree tree, String description) { 
		problems.add(new Problem(
				ANTLRUtils.location(source, tree), 
				description
			));
	}
}
