package org.simantics.graph.compiler.internal.templates;

import gnu.trove.map.hash.TIntIntHashMap;
import gnu.trove.map.hash.TObjectIntHashMap;

import java.util.Arrays;
import java.util.Collection;

import org.simantics.graph.compiler.ExternalFileLoader;
import org.simantics.graph.compiler.internal.store.LocationStore;
import org.simantics.graph.query.IGraph;
import org.simantics.graph.representation.External;
import org.simantics.graph.representation.Identity;
import org.simantics.graph.representation.Root;
import org.simantics.graph.representation.TransferableGraph1;
import org.simantics.graph.representation.Value;
import org.simantics.graph.store.GraphStore;
import org.simantics.graph.store.IdentityStore;
import org.simantics.graph.store.StatementStore;
import org.simantics.graph.store.ValueStore;
import org.simantics.ltk.Location;
import org.simantics.ltk.Problem;

public class GraphTemplate implements ITemplate {
	int[] prototypeResources;
	int[] parameterMapping;
	int[] statements;
	Value[] values;
	
	public GraphTemplate(GraphStore store, String[] parameters, TransferableGraph1 tg) {
		prototypeResources = new int[tg.resourceCount];
		Arrays.fill(prototypeResources, -1);
		
		this.values = tg.values;
		this.statements = tg.statements;
		this.parameterMapping = new int[parameters.length];
		TObjectIntHashMap<String> pmap = new TObjectIntHashMap<String>();
		for(int i=0;i<parameters.length;++i)
			pmap.put(parameters[i], i);
		
		for(Identity id : tg.identities) {
			if(id.definition instanceof Root) {
				Root def = (Root)id.definition;
				if(pmap.containsKey(def.name))
					parameterMapping[pmap.get(def.name)] = id.resource;
				else
					prototypeResources[id.resource] = store.identities.getRoot(def.name);
			}
			else if(id.definition instanceof External) {
				External def = (External)id.definition;
				prototypeResources[id.resource] =
					store.identities.getChild(
							prototypeResources[def.parent], 
							def.name);
			}
			else
				throw new RuntimeException("Invalid template");
		}
	}
	
	@Override
	public void apply(IGraph graph, GraphStore store, int[] parameters, ExternalFileLoader fileLoader, Collection<Problem> problems) {
		if(parameters.length != parameterMapping.length) {
			Location location = store.getStore(LocationStore.class)
				.getLocation(parameters[0]);
			problems.add(new Problem(location, "A template applied with wrong number of parameters."));
			return;
		}
		int[] resources = Arrays.copyOf(prototypeResources, prototypeResources.length);
		for(int i=0;i<parameterMapping.length;++i)
			resources[parameterMapping[i]] = parameters[i];
		
		IdentityStore identities = store.identities;
		for(int i=0;i<resources.length;++i)
			if(resources[i] < 0)
				resources[i] = identities.newResource();
		
		StatementStore statementStore = store.statements;
		for(int i=0;i<statements.length;i+=4) {
			statementStore.add(
					resources[statements[i]],
					resources[statements[i+1]],
					resources[statements[i+3]]
			);
		}
		
		ValueStore valueStore = store.values;
		for(Value value : values) {
			valueStore.setValue(resources[value.resource], value.value);
		}
	}

	public void map(TIntIntHashMap map) {
		for(int i=0;i<prototypeResources.length;++i) {
			int temp = prototypeResources[i];
			if(temp >= 0 && map.contains(temp))
				prototypeResources[i] = map.get(temp);
		}
	}	
}
