package org.simantics.export.core.util;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.simantics.NameLabelMode;
import org.simantics.NameLabelUtil;
import org.simantics.databoard.util.URIStringUtils;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.ParametrizedRead;
import org.simantics.db.common.request.UniqueRead;
import org.simantics.db.common.utils.traverser.TraverseQueryBuilder;
import org.simantics.db.common.utils.traverser.TraverseResult;
import org.simantics.db.exception.AssumptionException;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.ResourceNotFoundException;
import org.simantics.db.exception.ServiceException;
import org.simantics.db.exception.ValidationException;
import org.simantics.db.layer0.adapter.Instances;
import org.simantics.db.layer0.request.PossibleModel;
import org.simantics.db.request.Read;
import org.simantics.export.core.manager.Content;
import org.simantics.layer0.Layer0;

/**
 * Utility queries used by Exporting applications and user interfaces.
 *
 * @author toni.kalajainen@semantum.fi
 */
public class ExportQueries {
		
	public static boolean USE_INDEXER = true;
	
	public static Read<List<String>> toUris(final Collection<Resource> resources) {
		return new UniqueRead<List<String>>() {
			@Override
			public List<String> perform(ReadGraph graph) throws DatabaseException {
				List<String> result = new ArrayList<String>();
				for ( Resource r : resources ) {
					result.add( graph.getURI(r) );
				}
				return result;
			}
		};
	}
	
	public static Read<List<Resource>> toResources(final Collection<String> uris) {
		return new UniqueRead<List<Resource>>() {
			@Override
			public List<Resource> perform(ReadGraph graph) throws DatabaseException {
				List<Resource> result = new ArrayList<Resource>();
				for ( String uri : uris ) {
					result.add( graph.getResource(uri) );
				}
				
				return result;
			}
		};
	}
	
	public static Read<Resource> toResource(final String uri) {
		return new UniqueRead<Resource>() {
			@Override
			public Resource perform(ReadGraph graph) throws DatabaseException {
				return graph.getResource(uri);
			}
		};
	}
	
	public static Read<List<Resource>> toResources2(final Collection<Content> contents) {
		return new UniqueRead<List<Resource>>() {
			@Override
			public List<Resource> perform(ReadGraph graph) throws DatabaseException {
				List<Resource> result = new ArrayList<Resource>();
				for ( Content content : contents ) {
					result.add( graph.getResource( content.url ) );
				}
				
				return result;
			}
		};
	}
	
	/**
	 * Get query that returns labels for a collection of uris
	 * @param uris
	 * @return labels
	 */
	public static Read<Map<String, String>> labels(final Collection<String> uris) {
		return new UniqueRead<Map<String, String>>() {
			@Override
			public Map<String, String> perform(ReadGraph graph) throws DatabaseException {
				Map<String, String> result = new HashMap<String, String>();

				NameLabelMode mode = NameLabelUtil.getNameLabelMode(graph);

				for ( String uri : uris ) {
					String label = null;
            		try { 
           				Resource r = graph.getResource( uri );
           				label = NameLabelUtil.modalName(graph, r, mode);
            		} catch (AssumptionException e) {
            		} catch (ValidationException e) {
            		} catch (ServiceException e) {
            		}
            		
            		if ( label == null ) {
            			int c = uri.lastIndexOf('/');
            			if ( c>=0 ) label = uri.substring(c+1);
            		}
            		
            		if ( label==null || label.isEmpty() ) {
            			label = uri;
            		}
            		
       				result.put(uri, label);
				}
				
				return result;
			}
		};
	}

	/**
	 * Get query that returns a label for a uri
	 * @param uris
	 * @return labels
	 */
	public static Read<String> label(final String uri) {
		return new UniqueRead<String>() {
			@Override
			public String perform(ReadGraph graph) throws DatabaseException {
				String label = null;
           		try { 
          			Resource r = graph.getResource( uri );
       				label = NameLabelUtil.modalName(graph, r);
           		} catch (AssumptionException e) {
           		} catch (ValidationException e) {
           		} catch (ServiceException e) {
           		}
            		
           		if ( label == null ) {
           			int c = uri.lastIndexOf('/');
           			if ( c>=0 ) label = uri.substring(c+1);
           		}
           		
           		if ( label==null || label.isEmpty() ) {
           			label = uri;
           		}
           		
           		return label;
			}
		};
	}

	public static Read<String> label(final Resource r ) {
		return new UniqueRead<String>() {
			public String perform(ReadGraph graph) throws DatabaseException {				
				try {
       				return NameLabelUtil.modalName(graph, r);
           		} catch (AssumptionException e) {
           		} catch (ValidationException e) {
           		} catch (ServiceException e) {
           		}
				return null;
			}
		};
	}
	
	/**
	 * Get model resources from contents uris
	 * @param contentUris
	 * @return models
	 */
	public static Read<List<Resource>> toModels(final Collection<String> contentUris) {
		return new UniqueRead<List<Resource>>() {
			@Override
			public List<Resource> perform(ReadGraph graph) throws DatabaseException {
				List<Resource> result = new ArrayList<Resource>();
				for ( String uri : contentUris ) {
					Resource r = graph.getResource( uri );
					Resource possibleModel = graph.syncRequest( new PossibleModel(r) );
					if ( !result.contains( possibleModel ) ) result.add( possibleModel );
				}
				
				return result;
			}
		};		
	}	

	/**
	 * Get model resources of a content
	 * @param contentUris
	 * @return models
	 */
	public static Read<String> toModelUri(final String contentUri) {
		return new UniqueRead<String>() {
			@Override
			public String perform(ReadGraph graph) throws DatabaseException {
				Resource r = graph.getResource( contentUri );
				Resource possibleModel = graph.syncRequest( new PossibleModel(r) );
				return possibleModel == null ? null : graph.getURI(possibleModel);
			}
		};		
	}	

	public static ParametrizedRead<Collection<String>, Collection<String>> parametrizedInstancesOf(final String typeUri) {
		return new ParametrizedRead<Collection<String>, Collection<String>>() {
			@Override
			public Read<Collection<String>> get(Collection<String> parameter) {
				return instancesOf(parameter, typeUri);
			}
		};
	}

	/**
	 * Query instances of a type
	 * @param startingLocations
	 * @param type
	 * @return list or uris
	 */
	public static Read<Collection<String>> instancesOf(final Collection<String> startingLocations, final String typeUri)
	{
		return new UniqueRead<Collection<String>>() {
			public List<String> perform(ReadGraph graph) throws DatabaseException {
				if ( USE_INDEXER ) {					
					ArrayList<String> result = new ArrayList<String>();
					try {
						Resource type = graph.getResource( typeUri );
						Instances instances = graph.adapt(type, Instances.class);
						for ( String slUri : startingLocations ) {
							Resource sl = graph.getResource( slUri );
							
							Resource model = graph.syncRequest( new PossibleModel(sl) );
							if ( model == null ) model = sl;
							
							for ( Resource r : instances.find(graph, model) ) {
								String uri = graph.getPossibleURI( r );
								if ( uri != null
										&& uri.startsWith(slUri)
										&& uri.length() > slUri.length()
										&& uri.charAt(slUri.length()) == URIStringUtils.NAMESPACE_PATH_SEPARATOR)
								{
									result.add( uri );
								}
							}
						}
					} catch ( ResourceNotFoundException e ) {
						// If the type is not installed in the database, there sure are no instances.
						// This is not an error. The ontology just may not have been installed. The code must accept this.
					}
					return result;
				}				
				
				else 
					
				// Query implementation 
				{
					try {
						Layer0 L0 = Layer0.getInstance(graph);					
						Resource type = graph.getResource( typeUri );
						TraverseQueryBuilder builder = new TraverseQueryBuilder();
						builder.setStartResources( graph.syncRequest( ExportQueries.toResources(startingLocations) ) );
						builder.followRelation( L0.ConsistsOf );
						builder.followInstanceOf( L0.Library );
						
						try {
							builder.followInstanceOf( graph.getResource("http://www.apros.fi/Apros-6.1/Folder") );
						} catch ( ResourceNotFoundException e ) {}
						
						builder.followAndReturnInstanceOf( type );
						TraverseResult traverseResult = graph.syncRequest( builder.build() );
						return graph.syncRequest( ExportQueries.toUris( traverseResult.result ) );
					} catch ( ResourceNotFoundException e ) {
						// If the type is not installed in the database, there sure are no instances.
						// This is not an error. The ontology just may not have been installed. The code must accept this.
						return Collections.emptyList();
					}
				}
								
			}
		};
	}
	
	/**
	 * Query instances of a type
	 * @param startingLocations
	 * @param type
	 * @return list or uris
	 */
	public static Read<List<Resource>> instancesOf(final Collection<Resource> startingLocations, final Resource type)
	{
		return new UniqueRead<List<Resource>>() {
			public List<Resource> perform(ReadGraph graph) throws DatabaseException {
				Instances instances = graph.adapt(type, Instances.class);
				ArrayList<Resource> result = new ArrayList<Resource>();
				for ( Resource sl : startingLocations ) {
					result.addAll( instances.find(graph, sl) );
				}
				return result;
			}
		};
	}
	
	
}
