/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.editors.win32;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.dynamichelpers.ExtensionTracker;
import org.eclipse.core.runtime.dynamichelpers.IExtensionChangeHandler;
import org.eclipse.core.runtime.dynamichelpers.IExtensionTracker;
import org.eclipse.core.runtime.dynamichelpers.IFilter;
import org.simantics.editors.win32.ole.EditorDefinition;
import org.simantics.utils.ui.ErrorLogger;

public class EditorDefinitionManager implements IExtensionChangeHandler {

    private final static String                           NAMESPACE  = Activator.PLUGIN_ID;

    private final static String                           EP_NAME    = "editorDefinition";

    private ExtensionTracker                              tracker;

    private List<IEditorDefinitionExtension> extensions = new ArrayList<IEditorDefinitionExtension>();

    private static EditorDefinitionManager       INSTANCE;

    public static synchronized EditorDefinitionManager getInstance() {
        if (INSTANCE == null)
            INSTANCE = new EditorDefinitionManager();
        return INSTANCE;
    }

    public static synchronized void dispose() {
        if (INSTANCE != null) {
            INSTANCE.close();
            INSTANCE = null;
        }
    }

    private EditorDefinitionManager() {
        tracker = new ExtensionTracker();

        // Cache defined actions
        IExtensionPoint expt = Platform.getExtensionRegistry().getExtensionPoint(NAMESPACE, EP_NAME);
        loadExtensions(expt.getConfigurationElements());

        // Start tracking for new and removed extensions
        IFilter filter = ExtensionTracker.createExtensionPointFilter(expt);
        tracker.registerHandler(this, filter);
    }

    private void close() {
        tracker.close();
        tracker = null;
        extensions = new ArrayList<IEditorDefinitionExtension>();
    }

    public synchronized List<IEditorDefinitionExtension> getExtensions(String perspectiveId) {
        if (extensions == null)
            return Arrays.asList();
        return Collections.unmodifiableList(extensions);
    }

    private synchronized void loadExtensions(IConfigurationElement[] elements) {
        for (IConfigurationElement el : elements) {
        	try {
        		EditorDefinition fac = (EditorDefinition)el.createExecutableExtension("factory");
            
        		IEditorDefinitionExtension ext = new IEditorDefinitionExtension.Stub(fac);

            	// Start tracking the new extension object, its removal will be notified of
            	// with removeExtension(extension, Object[]).
            	tracker.registerObject(el.getDeclaringExtension(), ext, IExtensionTracker.REF_STRONG);

            	extensions.add(ext);
        	} catch (Exception e) {
        		ErrorLogger.defaultLogError(e);
        	}
        }
    }


    @Override
    public void addExtension(IExtensionTracker tracker, IExtension extension) {
        loadExtensions(extension.getConfigurationElements());
    }

    @Override
    public synchronized void removeExtension(IExtension extension, Object[] objects) {
        List<IEditorDefinitionExtension> newExtensions = new ArrayList<IEditorDefinitionExtension>();

        for (Object o : objects) {
            IEditorDefinitionExtension ext = (IEditorDefinitionExtension) o;
            tracker.unregisterObject(extension, ext);
            extensions.remove(ext);
        }

        // Atomic assignment
        this.extensions = newExtensions;
    }
    
    public EditorDefinition getFactoryForSuffix(String suffix) {
    	for (IEditorDefinitionExtension e : extensions) {
    		if (e.getFactory().supportSuffix(suffix)) {
    			return e.getFactory();
    		}
    		
    	}
    	return null;    	
    }

}
