/*******************************************************************************
 * Copyright (c) 2007, 2013 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - index based searching and graph manipulation (#4255)
 *******************************************************************************/
package org.simantics.debug.ui;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.ActionContributionItem;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.ToolBar;
import org.eclipse.ui.ISharedImages;
import org.eclipse.ui.PlatformUI;
import org.simantics.Simantics;
import org.simantics.db.exception.DatabaseException;
import org.simantics.debug.ui.internal.Activator;
import org.simantics.debug.ui.internal.DebugUtils;
import org.simantics.ui.SimanticsUI;
import org.simantics.ui.workbench.ResourceEditorPart;
import org.simantics.utils.ui.BundleUtils;
import org.simantics.utils.ui.LayoutUtils;

public class GraphDebuggerEditor extends ResourceEditorPart {

    public static final String EDITOR_ID = "org.simantics.debug.graphDebuggerEditor";

    private GraphDebugger      debugger;
    private IAction            backAction;
    private IAction            forwardAction;
    private IAction            refreshAction;
    private IAction            findAction;
    private IAction            addStatementAction;
    private IAction            addResourceAction;

    @Override
    public void createPartControl(Composite parent) {
        // Create UI
        parent.setLayout(LayoutUtils.createNoBorderGridLayout(1));

        debugger = new GraphDebugger(parent, SWT.NONE, getSession(), getInputResource(), getSite());
        debugger.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
        debugger.setLayout(LayoutUtils.createNoBorderGridLayout(1));

        Composite bar = new Composite(debugger, SWT.NONE);
        bar.setLayout(new GridLayout(3, false));
        bar.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));

        backAction = new BackAction();
        forwardAction = new ForwardAction();
        refreshAction = new RefreshAction();
        findAction = new FindAction();
        addStatementAction = new AddStatementAction();
        addResourceAction = new AddResourceAction();

        ToolBar toolBar = new ToolBar(bar, SWT.HORIZONTAL | SWT.FLAT);

        new ActionContributionItem(backAction).fill(toolBar, 0);
        new ActionContributionItem(forwardAction).fill(toolBar, 1);
        new Separator().fill(toolBar, 2);
        new ActionContributionItem(refreshAction).fill(toolBar, 3);
        new Separator().fill(toolBar, 4);
        new ActionContributionItem(findAction).fill(toolBar, 5);
        new ActionContributionItem(addStatementAction).fill(toolBar, 6);
        new ActionContributionItem(addResourceAction).fill(toolBar, 7);

        debugger.createResourceText(bar);
        Control dropLabel = debugger.createDropLabel(bar);
        GridDataFactory.fillDefaults().grab(true, false).align(SWT.FILL, SWT.FILL).hint(SWT.DEFAULT, 20).span(3, 1).applyTo(dropLabel);
        Control browser = debugger.createBrowser(debugger);
        GridDataFactory.fillDefaults().grab(true, true).span(3, 1).applyTo(browser);

        debugger.addHistoryListener(new GraphDebugger.HistoryListener() {
            @Override
            public void historyChanged() {
                updateActionStates();
            }
        });

        updateActionStates();
    }

    @Override
    public void setFocus() {
        if (debugger != null)
            debugger.setFocus();
    }

    public void back() {
        debugger.back();
    }

    public void forward() {
        debugger.forward();
    }

    public void refreshBrowser() {
        debugger.refreshBrowser();
    }

    private void updateActionStates() {
        backAction.setEnabled(!debugger.hasBackHistory());
        forwardAction.setEnabled(!debugger.hasForwardHistory());
    }

    class RefreshAction extends Action {
        public RefreshAction() {
            super("Refresh", BundleUtils.getImageDescriptorFromPlugin(SimanticsUI.PLUGIN_ID, "icons/refresh.gif"));
            setToolTipText("Refresh");
        }
        @Override
        public void run() {
            refreshBrowser();
        }
    }

    class FindAction extends Action {
        public FindAction() {
            super("Find", BundleUtils.getImageDescriptorFromPlugin(Activator.PLUGIN_ID, "icons/cog_blue.png"));
            setToolTipText("Find Resource");
        }
        @Override
        public void run() {
            DebugUtils.find(Simantics.getSession(), debugger);
        }
    }

    class AddStatementAction extends Action {
        public AddStatementAction() {
            super("AddStatement", BundleUtils.getImageDescriptorFromPlugin(Activator.PLUGIN_ID, "icons/cog_add.png"));
            setToolTipText("Add Statement Between Existing Resources");
        }
        @Override
        public void run() {
            try {
                DebugUtils.addStatement(Simantics.getSession(), debugger);
            } catch (DatabaseException e) {
                Activator.getDefault().getLog().log(new Status(IStatus.ERROR, Activator.PLUGIN_ID, e.getMessage(), e));
            }
        }
    }

    class AddResourceAction extends Action {
        public AddResourceAction() {
            super("AddResource", BundleUtils.getImageDescriptorFromPlugin(Activator.PLUGIN_ID, "icons/cog_add.png"));
            setToolTipText("Add New Related Resource");
        }
        @Override
        public void run() {
            try {
                DebugUtils.addResource(Simantics.getSession(), debugger);
            } catch (DatabaseException e) {
                Activator.getDefault().getLog().log(new Status(IStatus.ERROR, Activator.PLUGIN_ID, e.getMessage(), e));
            }
        }
    }

    class BackAction extends Action {
        public BackAction() {
            super("Back", Action.AS_PUSH_BUTTON);
            setToolTipText("Back");
            setImageDescriptor(PlatformUI.getWorkbench().getSharedImages().getImageDescriptor(ISharedImages.IMG_TOOL_BACK));
            setDisabledImageDescriptor(PlatformUI.getWorkbench().getSharedImages().getImageDescriptor(ISharedImages.IMG_TOOL_BACK_DISABLED));
        }
        @Override
        public void run() {
            back();
            updateActionStates();
        }
    }

    class ForwardAction extends Action {
        public ForwardAction() {
            super("Forward", Action.AS_PUSH_BUTTON);
            setToolTipText("Forward");
            setImageDescriptor(PlatformUI.getWorkbench().getSharedImages().getImageDescriptor(ISharedImages.IMG_TOOL_FORWARD));
            setDisabledImageDescriptor(PlatformUI.getWorkbench().getSharedImages().getImageDescriptor(ISharedImages.IMG_TOOL_FORWARD_DISABLED));
        }
        @Override
        public void run() {
            forward();
            updateActionStates();
        }
    }

}
