/*******************************************************************************
 * Copyright (c) 2016 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     THTH ry - initial API and implementation
 *******************************************************************************/
package org.simantics.debug.browser.utils;

import java.util.Arrays;
import java.util.Collection;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Statement;
import org.simantics.db.VirtualGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.service.VirtualGraphSupport;

public class StatementInfo implements Comparable<StatementInfo> {
    public final ResourceInfo subject; // null, if not asserted
    public final ResourceInfo object;
    public final ResourceInfo[] objectTypes;
    public final String graph; // null, if in DB
    
    public StatementInfo(ReadGraph graph, Resource defaultSubject, Statement statement) throws DatabaseException {
        Resource subject = statement.getSubject();
        Resource predicate = statement.getPredicate();
        Resource object = statement.getObject();
        Collection<Resource> objectTypes = graph.getPrincipalTypes(object);
        
        if(subject.equalsResource(defaultSubject))
            this.subject = null;
        else
            this.subject = new ResourceInfo(graph, subject);
        this.object = new ResourceInfo(graph, object);
        if(objectTypes.isEmpty())
            this.objectTypes = null;
        else
            this.objectTypes = mapTypes(graph, objectTypes);
        this.graph = getGraphName(graph, subject, predicate, object);
    }
    

    private static ResourceInfo[] mapTypes(ReadGraph graph, Collection<Resource> types) throws DatabaseException {
        ResourceInfo[] result = new ResourceInfo[types.size()];
        int i=0;
        for(Resource type : types)
            result[i++] = new ResourceInfo(graph, type);
        Arrays.sort(result);
        return result;
    }


    public static String getGraphName(ReadGraph graph, Resource s, Resource p, Resource o) throws DatabaseException {
        VirtualGraphSupport vgs = graph.getService(VirtualGraphSupport.class);
        VirtualGraph vg = vgs.getGraph(graph, s, p, o);
        if(vg != null)
            return vg.toString();
        else
            return null;
    }

    @Override
    public int compareTo(StatementInfo o) {
        return object.compareTo(o.object);
    }
}
