/*******************************************************************************
 * Copyright (c) 2016 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     THTH ry - initial API and implementation
 *******************************************************************************/
package org.simantics.debug.browser.utils;

import org.simantics.databoard.Bindings;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.type.Datatype;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.layer0.Layer0;
import org.simantics.utils.strings.AlphanumComparator;

public class ResourceInfo implements Comparable<ResourceInfo> {
    public final String label;
    public final String uri;
    public final long resourceId;
    public final Resource resource;
    
    public ResourceInfo(ReadGraph graph, Resource resource) throws DatabaseException {
        this.label = getLabel(graph, resource);
        this.uri = graph.getPossibleURI(resource);
        this.resourceId = resource.getResourceId();
        this.resource = resource;
    }
    
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(80);
        sb.append("<a href=\"")
          .append(resourceId)
          .append("\"");
        
        if (uri != null)
            sb.append(" title=\"").append(uri).append("\"");
        
        sb.append(">")
          .append(Escapes.html(label))
          .append("</a>");
        
        return sb.toString();
    }
    
    public static String getLabel(ReadGraph graph, Resource resource) throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(graph);
        String name = graph.getPossibleRelatedValue(resource, L0.HasName);
        if(name != null) {
            if(name.equals("Inverse")) {
                Resource inverse = graph.getPossibleInverse(resource);
                Resource parent = graph.getPossibleObject(resource, L0.PartOf);
                if(inverse != null && inverse.equals(parent)) {
                    String parentName = graph.getPossibleRelatedValue(parent, L0.HasName);
                    if(parentName != null)
                        return parentName + "/Inverse";
                }
            }
            return name;
        }
        
        if(graph.hasValue(resource)) {
            Datatype datatype = graph.getDataType(resource);
            Binding binding = Bindings.getBinding(datatype);
            Object value = graph.getValue(resource, binding);
            try {
                String valueString = binding.toString(value, true);
                if(valueString.length() < 100)
                    return valueString;
                else
                    return valueString.substring(0, 100) + "...";
            } catch (BindingException e) {
                e.printStackTrace();
            }
        }
        
        return "$" + resource.getResourceId();
    }

    @Override
    public int compareTo(ResourceInfo o) {
        return AlphanumComparator.CASE_INSENSITIVE_COMPARATOR.compare(label, o.label);
    }
}
