/*******************************************************************************
 * Copyright (c) 2016 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     THTH ry - initial API and implementation
 *******************************************************************************/
package org.simantics.debug.browser.sections;

import gnu.trove.map.hash.THashMap;
import gnu.trove.map.hash.TObjectIntHashMap;

import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Map;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Statement;
import org.simantics.db.exception.DatabaseException;
import org.simantics.debug.browser.utils.ResourceInfo;
import org.simantics.debug.browser.utils.StatementInfo;
import org.simantics.layer0.Layer0;

public class RawStatementsSection implements ResourceBrowserSection {
    public Resource resource;
    public THashMap<Resource, ArrayList<Statement>> statementsByPredicates =
            new THashMap<Resource, ArrayList<Statement>>();

    @Override
    public double getPriority() {
        return 100.0;
    }

    @Override
    public void toHtml(ReadGraph graph, PrintWriter out) throws DatabaseException {
        THashMap<Resource, THashMap<Resource, ArrayList<Statement>>> predicateClassification = 
                new THashMap<Resource, THashMap<Resource, ArrayList<Statement>>>();
        
        Layer0 L0 = Layer0.getInstance(graph);
        for(Map.Entry<Resource, ArrayList<Statement>> entry
                : statementsByPredicates.entrySet()) {
            Resource domain = getDomain(graph, L0, resource, entry.getKey());
            THashMap<Resource, ArrayList<Statement>> map = predicateClassification.get(domain);
            if(map == null) {
                map = new THashMap<Resource, ArrayList<Statement>>();
                predicateClassification.put(domain, map);
            }
            map.put(entry.getKey(), entry.getValue());
        }
        
        ArrayList<Resource> domains = new ArrayList<Resource>(predicateClassification.keySet());
        Collections.sort(domains, new TypeComparator(graph, L0));
        
        for(Resource domain : domains) {
            out.println("<h2>" + new ResourceInfo(graph, domain) + "</h2>");
            generateStatementTable(graph, out, predicateClassification.get(domain));
        }
    }
    
    private static class TypeComparator implements Comparator<Resource> {
        private final ReadGraph graph;
        private final Layer0 L0;
        private TObjectIntHashMap<Resource> depthCache = new TObjectIntHashMap<Resource>();
        
        public TypeComparator(ReadGraph graph, Layer0 L0) {
            this.graph = graph;
            this.L0 = L0;
            depthCache.put(L0.Entity, 0);
        }

        private int depth(Resource type) {
            if(depthCache.containsKey(type))
                return depthCache.get(type);
            
            depthCache.put(type, 0); // loop guard
            try {
                int depth = 0;
                for(Resource superType : graph.getObjects(type, L0.Inherits))
                    depth = Math.max(depth, depth(superType));
                ++depth;
                depthCache.put(type, depth);
                return depth;
            } catch(DatabaseException e) {
                e.printStackTrace();
                return 0;
            }
        }
        
        @Override
        public int compare(Resource type1, Resource type2) {
            return Integer.compare(depth(type2), depth(type1));
        }
    }
    
    public void generateStatementTable(ReadGraph graph, PrintWriter out,
            THashMap<Resource, ArrayList<Statement>> statementsByPredicates) 
                    throws DatabaseException {
        ArrayList<ResourceInfo> predicateInfos = new ArrayList<ResourceInfo>(statementsByPredicates.size());
        for(Resource predicate : statementsByPredicates.keySet())
            predicateInfos.add(new ResourceInfo(graph, predicate));
        Collections.sort(predicateInfos);

        if (!predicateInfos.isEmpty()) {
            out.println("<div id=\"rawStatementContent\">");
            out.println("<table>");
            out.println("<tr><th>Predicate</th><th>Object</th><th>Notes</th></tr>");
            for(ResourceInfo predicateInfo : predicateInfos) {
                ArrayList<Statement> statements = statementsByPredicates.get(predicateInfo.resource);
                ArrayList<StatementInfo> statementInfos = new ArrayList<StatementInfo>(statements.size());
                for(Statement statement : statements)
                    statementInfos.add(new StatementInfo(graph, resource, statement));
                Collections.sort(statementInfos);
                
                boolean first = true;
                for(StatementInfo statementInfo : statementInfos) {
                    out.println("\t<tr>");
                    if(first) {
                        out.println("\t\t<td rowspan=\""+statementInfos.size()+"\">" + predicateInfo + "</td>");
                        first = false;
                    }
                    out.print("\t\t<td");
                    if (statementInfo.subject != null)
                        out.print(" class=\"asserted\"");
                    out.print(">" + statementInfo.object);
                    if(statementInfo.objectTypes != null) {
                        out.print("<span class=\"resourceType\"> : ");
                        for(int i=0;i<statementInfo.objectTypes.length;++i) {
                            if(i > 0)
                                out.print(", ");
                            out.print(statementInfo.objectTypes[i]);
                        }
                        out.print("</span>");
                    }
                    out.println("</td>");
                    if(statementInfo.graph != null || statementInfo.subject != null) {
                        out.print("\t\t<td>");
                        if(statementInfo.graph != null)
                            out.print(" from graph " + statementInfo.graph);
                        if(statementInfo.subject != null)
                            out.print(" from type " + statementInfo.subject);
                        out.println("</td>");
                    }
                    out.println("\t</tr>");
                }
            }
            out.println("</table>");
            out.println("</div>");
        }
    }
    
    public static Resource getDomain(ReadGraph graph, Layer0 L0, Resource subject, Resource predicate) throws DatabaseException {
        Collection<Resource> domains = graph.getObjects(predicate, L0.HasDomain);
        switch(domains.size()) {
        case 0: return L0.Entity;
        case 1: return domains.iterator().next();
        }
        
        for(Resource domain : domains)
            if(graph.isInstanceOf(subject, domain))
                return domain;

        return domains.iterator().next();
    }

    /*
        out.println("<div id=\"rawStatementContent\">");
        out.println("<script type=\"text/javascript\">");
        out.println("$(document).ready(function() {");
        out.println("\t$('#rawStatementContent').html( '<table id=\"rawStatementTable\" cellpadding=\"0\" cellspacing=\"0\" border=\"0\" class=\"tablesorter dataTables\"></table>' );");
        out.println("\tvar oTable = $('#rawStatementTable')");
        out.println("\t\t.dataTable({");
        out.println("\t\t\t\"stateSave\": true,");
        out.println("\t\t\t\"jQueryUI\": true,");
        out.println("\t\t\t\"paging\": false,");
        out.println("\t\t\t\"pagingType\": \"full_numbers\",");
        out.println("\t\t\t\"filter\": true,");
        out.println("\t\t\t\"info\": true,");
        //out.println("\t\t\t\"dom\": '<\"H\"lr>t<\"F\"ip>',");

        out.println("\t\t\t\"data\" : [");
        for (Statement stm : statements) {
            out.println("\t\t\t\t[");
            out.println("\t\t\t\t\t'" + toLinkedLabel(graph, stm.getSubject()) + "',");
            out.println("\t\t\t\t\t'" + toLinkedLabel(graph, stm.getPredicate()) + "',");
            out.println("\t\t\t\t\t'" + toLinkedLabel(graph, stm.getObject()) + "'");
            out.println("\t\t\t\t],");
        }
        out.println("\t\t\t],");

        out.println("\t\t\t\"columns\": [");
        out.println("\t\t\t\t{ \"sTitle\" : \"Subject\", \"sClass\": \"result\" },");
        out.println("\t\t\t\t{ \"sTitle\" : \"Predicate\", \"sClass\": \"result\" },");
        out.println("\t\t\t\t{ \"sTitle\" : \"Object\", \"sClass\": \"result\" }");
        out.println("\t\t\t],");

        out.println("\t\t\t\"oLanguage\": {");
        out.println("\t\t\t\t\"oPaginate\": {");
        out.println("\t\t\t\t\t\"sPrevious\": \"<\",");
        out.println("\t\t\t\t\t\"sNext\": \">\",");
        out.println("\t\t\t\t\t\"sLast\": \">>\",");
        out.println("\t\t\t\t\t\"sFirst\": \"<<\"");
        out.println("\t\t\t\t},");
        out.println("\t\t\t\t\"sSearch\": \"Filter:\"");
        out.println("\t\t\t},");

        out.println("\t\t\t\"oSearch\": {");
        out.println("\t\t\t\t\"sSearch\": \"\"");
        out.println("\t\t\t}");

        out.println("\t\t});");
        out.println("\t});");
        out.println("</script>");
        out.println("</div>");
     */
}
