/*******************************************************************************
 * Copyright (c) 2016 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     THTH ry - initial API and implementation
 *******************************************************************************/
package org.simantics.debug.browser.internal.rewriters;

import java.util.ArrayList;

import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Statement;
import org.simantics.db.exception.DatabaseException;
import org.simantics.debug.browser.content.ResourceBrowserContent;
import org.simantics.debug.browser.content.ResourceBrowserRewriter;
import org.simantics.debug.browser.sections.BreadcrumbSection;
import org.simantics.debug.browser.sections.RawStatementsSection;
import org.simantics.debug.browser.sections.BreadcrumbSection.Node;
import org.simantics.layer0.Layer0;

public enum BreadcrumbCreator implements ResourceBrowserRewriter {
    INSTANCE;

    @Override
    public void rewrite(ReadGraph graph, ResourceBrowserContent content)
            throws DatabaseException {
        ArrayList<Node> nodes = new ArrayList<Node>();
        collectBreadcrumbs(graph, nodes, content.resource);
        if(!nodes.isEmpty()) {
            content.putSection(BreadcrumbSection.class,
                    new BreadcrumbSection(nodes.toArray(new Node[nodes.size()])));
            
            Layer0 L0 = Layer0.getInstance(graph);
            RawStatementsSection rawStatementsSection = content.getSection(RawStatementsSection.class);
            rawStatementsSection.statementsByPredicates.remove(L0.HasName);
            rawStatementsSection.statementsByPredicates.remove(L0.PartOf);
        }
    }

    private static void collectBreadcrumbs(ReadGraph graph, ArrayList<Node> nodes,
            Resource resource) throws DatabaseException {
        if(resource.equals(graph.getRootLibrary())) {
            nodes.add(new Node("", "http:/", resource.getResourceId()));
            return;
        }
        Layer0 L0 = Layer0.getInstance(graph);
        Resource parent = graph.getPossibleObject(resource, L0.PartOf);
        if(parent != null) {
            collectBreadcrumbs(graph, nodes, parent);
            String name = graph.getPossibleRelatedValue(resource, L0.HasName, Bindings.STRING);
            if(name != null)
                nodes.add(new Node("/", name, resource.getResourceId()));
            else
                nodes.add(new Node("/", String.valueOf(resource.getResourceId()), resource.getResourceId()));
            return;
        }
        Statement propertyInverseStatement = graph.getPossibleStatement(resource, L0.PropertyOf);
        if(propertyInverseStatement != null) {
            collectBreadcrumbs(graph, nodes, propertyInverseStatement.getObject());
            String name = graph.getPossibleRelatedValue(graph.getInverse(propertyInverseStatement.getPredicate()),
                    L0.HasName, Bindings.STRING);
            nodes.add(new Node("#", name, resource.getResourceId()));
            return;
        }
        Resource assertion = graph.getPossibleObject(resource, L0.HasObjectInverse);
        if(assertion != null) {
            parent = graph.getPossibleObject(assertion, L0.Asserts_Inverse);
            if(parent != null) {
                collectBreadcrumbs(graph, nodes, parent);
                Resource predicate = graph.getPossibleObject(assertion, L0.HasPredicate);
                String name = predicate == null ? null 
                        : (String)graph.getPossibleRelatedValue(predicate, L0.HasName, Bindings.STRING);
                if(name != null)
                    nodes.add(new Node("#assertion#", name, resource.getResourceId()));
                else
                    nodes.add(new Node("#assertion#", String.valueOf(resource.getResourceId()), resource.getResourceId()));
                return;
            }
        }
    }

    @Override
    public double getPriority() {
        return 1;
    }
}
