/*******************************************************************************
 * Copyright (c) 2007, 2012 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.swt.widgets.impl;

import org.eclipse.jface.viewers.ISelection;
import org.simantics.db.ReadGraph;
import org.simantics.db.common.request.UnaryRead;
import org.simantics.db.common.utils.Logger;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.management.ISessionContext;
import org.simantics.db.procedure.Listener;
import org.simantics.utils.ReflectionUtils;
import org.simantics.utils.datastructures.Pair;
import org.simantics.utils.ui.ISelectionUtils;

/**
 * @author Antti Villberg
 *
 * @param <Input> expected and checked input type
 * @param <Output> result type of factory
 */
public abstract class ReadFactoryImpl<Input, Output> implements ReadFactory<Input, Output> {

    final Class<?> inputClass;
    final boolean sync;

    protected ReadFactoryImpl() {
    	this(false);
    }
    
    protected ReadFactoryImpl(boolean sync) {
    	inputClass = ReflectionUtils.getSingleParameterType(getClass());
    	this.sync = sync;
    }

    public Object getIdentity(Object inputContents) {
        return new Pair<Object, Class<?>>(inputContents, getClass());
    }

    private Object getInputContents(Object input, Class<?> inputClass) {
        if (inputClass.isInstance(input))
            return input;
        if (input instanceof ISelection)
            return ISelectionUtils.filterSingleSelection(input, inputClass);
        return null;
    }

    @Override
    public void listen(ISessionContext context, Object input, Listener<Output> listener) {
        final Object inputContents = getInputContents(input, inputClass);
        if (inputContents != null) {
        	UnaryRead<Object, Output> read = new UnaryRead<Object, Output>(getIdentity(inputContents)) {
                @SuppressWarnings("unchecked")
                @Override
                public Output perform(ReadGraph graph) throws DatabaseException {
                    return ReadFactoryImpl.this.perform(graph, (Input) inputContents);
                }
                @Override
                public String toString() {
                	return getIdentity(inputContents).toString();
                }
            };
            if(sync)
				try {
					Output out = context.getSession().syncRequest(read, listener);
					listener.execute(out);
				} catch (DatabaseException e) {
					Logger.defaultLogError(e);
				}
			else
            	context.getSession().asyncRequest(read, listener);
        }
    }

    abstract public Output perform(ReadGraph graph, Input input) throws DatabaseException;

}
