/*******************************************************************************
 * Copyright (c) 2007, 2012 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.swt.widgets;

import org.eclipse.mylyn.wikitext.core.parser.MarkupParser;
import org.eclipse.swt.browser.Browser;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.simantics.Simantics;
import org.simantics.browsing.ui.common.ErrorLogger;
import org.simantics.browsing.ui.swt.widgets.impl.ReadFactory;
import org.simantics.browsing.ui.swt.widgets.impl.WidgetSupport;
import org.simantics.db.ReadGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.management.ISessionContext;
import org.simantics.db.procedure.Listener;
import org.simantics.db.request.Read;
import org.simantics.utils.datastructures.map.Tuple;
import org.simantics.wiki.ui.SimanticsDialect;
import org.simantics.wiki.ui.language.MediaWikiLanguage;

public class WikiBrowser extends WidgetImpl {

    private ReadFactory<?, String>   textFactory;
    private ReadFactory<?, Variable> variableFactory;

    private String                   text;
    private Variable                 variable;

    private final Display            display;
    private final Browser   browser;

    private Tuple                    lastAppliedParametrization;

    public WikiBrowser(Composite parent, WidgetSupport support, int style) {
    	super(support);
        display = parent.getDisplay();
        browser = new Browser(parent, style);
        support.register(this);
    }

    public void setTextFactory(ReadFactory<?, String> textFactory) {
        this.textFactory = textFactory;
    }

    public void setVariableFactory(ReadFactory<?, Variable> variableFactory) {
        this.variableFactory = variableFactory;
    }

    public Browser getWidget() {
        return browser;
    }

    @Override
    public Control getControl() {
        return browser;
    }

    private void update() {
        // Prevent synchronization problems since multiple threads and context
        // switches are involved in this process
        String text = this.text;
        Variable variable = this.variable;
        System.out.println("update(" + text + ", " + variable + ")");
        if (text == null || variable == null || browser.isDisposed())
            return;

        display.asyncExec(new Runnable() {

            @Override
            public void run() {
                final String text = WikiBrowser.this.text;
                final Variable variable = WikiBrowser.this.variable;
                if (text == null || variable == null || browser.isDisposed()) {
                    return;
                }

                Tuple checkParam = new Tuple(text, variable);
                if (checkParam.equals(lastAppliedParametrization))
                    return;
                lastAppliedParametrization = checkParam;

                try {
                    String markup = Simantics.getSession().syncRequest(new Read<String>() {
                        @Override
                        public String perform(ReadGraph graph) throws DatabaseException {
                            return SimanticsDialect.INSTANCE.apply(graph, variable, text);
                        }
                    });

                    MarkupParser markupParser = new MarkupParser();
                    MediaWikiLanguage language = new MediaWikiLanguage();
                    markupParser.setMarkupLanguage(language);
                    final String htmlContent = markupParser.parseToHtml(markup);
                    if (htmlContent == null)
                        return;

                    browser.setText(htmlContent);

                } catch (DatabaseException e) {
                    ErrorLogger.defaultLogError(e);
                }
            }
        });
    }

    @Override
    public void setInput(ISessionContext context, Object input) {

        if(textFactory != null) {
            textFactory.listen(context, input, new Listener<String>() {

                @Override
                public void exception(Throwable t) {
                    ErrorLogger.defaultLogError(t);
                }

                @Override
                public void execute(final String text) {
                    WikiBrowser.this.text = text;
                    update();
                }

                @Override
                public boolean isDisposed() {
                    return browser.isDisposed();
                }

            });
        }

        if(variableFactory != null) {
            variableFactory.listen(context, input, new Listener<Variable>() {

                @Override
                public void exception(Throwable t) {
                    ErrorLogger.defaultLogError(t);
                }

                @Override
                public void execute(final Variable variable) {
                    WikiBrowser.this.variable = variable;
                    update();
                }

                @Override
                public boolean isDisposed() {
                    return browser.isDisposed();
                }

            });

        }

    }

    public boolean isDisposed() {
        return browser.isDisposed();
    }

}
