/*******************************************************************************
 * Copyright (c) 2007, 2012 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.swt.widgets;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.IWorkbenchSite;
import org.simantics.browsing.ui.NodeContext;
import org.simantics.browsing.ui.common.ErrorLogger;
import org.simantics.browsing.ui.model.InvalidContribution;
import org.simantics.browsing.ui.model.browsecontexts.BrowseContext;
import org.simantics.browsing.ui.model.dnd.DndBrowseContext;
import org.simantics.browsing.ui.swt.widgets.impl.WidgetSupport;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.ResourceNotFoundException;
import org.simantics.db.procedure.Procedure;
import org.simantics.db.request.Read;
import org.simantics.ui.SimanticsUI;
import org.simantics.utils.ui.ExceptionUtils;

public class ModelBrowser extends GraphExplorerComposite {

    volatile DndBrowseContext dndBrowseContext;

    private static final String CONTEXT_MENU_ID = "#GraphExplorerPopup";

    protected Set<String>                                 propertyBrowseContexts = Collections.emptySet();

    protected boolean                                     hideComparatorSelector = false;
    protected boolean                                     hideViewpointSelector = false;
    protected boolean                                     hideFilter = false;

    @Override
    protected String getContextMenuId() {
    	if(contextMenuId != null) return contextMenuId;
    	else return CONTEXT_MENU_ID;
    }

    @Override
    protected void createControls(Composite parent) {
        this.propertyBrowseContexts = loadBrowseContexts(getBrowseContexts());
        super.createControls(parent);
    }

    protected static Set<String> loadBrowseContexts(final Set<String> browseContexts) {
        try {
            return SimanticsUI.getSession().syncRequest(new Read<Set<String>>() {
                @Override
                public Set<String> perform(ReadGraph graph) throws DatabaseException {
                    Collection<Resource> browseContextResources = new ArrayList<Resource>(browseContexts.size());
                    for (String browseContext : browseContexts) {
                        try {
                            browseContextResources.add(graph.getResource(browseContext));
                        } catch (ResourceNotFoundException e) {
                            // Expected result, if no modelled contributions exist.
                            //System.err.println("Didn't find " + browseContext + " while loading model browser.");
                        } catch (DatabaseException e) {
                            ExceptionUtils.logError("Didn't find " + browseContext + " while loading model browser.", e);
                        }
                    }
                    Collection<Resource> allBrowseContextResources = BrowseContext.findSubcontexts(graph, browseContextResources);
                    Set<String> result = new HashSet<String>();
                    for (Resource r : allBrowseContextResources)
                        result.add(graph.getURI(r));
                    return result;
                }
            });
        } catch (DatabaseException e) {
            ExceptionUtils.logAndShowError("Failed to load modeled browse contexts for property page, see exception for details.", e);
            return browseContexts;
        }
    }
    
    public ModelBrowser(final Set<String> _browseContexts, Map<String, Object> args, IWorkbenchSite site, Composite parent, WidgetSupport support, int style) {
    	
    	super(args, site, parent, support, style);
    
    	setBrowseContexts(_browseContexts);
    	
        // As long as these are not useful, don't shown them by default.
        this.hideComparatorSelector = true;
        this.hideViewpointSelector = true;

        SimanticsUI.getSession().asyncRequest(new ReadRequest() {
            @Override
            public void run(ReadGraph graph) throws DatabaseException { 
                ArrayList<Resource> browseContexts = new ArrayList<Resource>();
                for (String uri : _browseContexts) {
                    Resource browseContext = graph.getPossibleResource(uri);
                    if (browseContext != null)
                        browseContexts.add(browseContext);
                }
                try {
                    dndBrowseContext = DndBrowseContext.create(graph, browseContexts);
                } catch (InvalidContribution e) {
                    ErrorLogger.defaultLogError(e);
                }
            }
        });
    }

    @Override
    protected void handleDrop(final Object data, final NodeContext target) {
        if (target == null)
            return;

        SimanticsUI.getSession().asyncRequest(new Read<Runnable>() {
            @Override
            public Runnable perform(ReadGraph graph) throws DatabaseException {
                if (dndBrowseContext == null)
                    return null;
                return dndBrowseContext.getAction(graph, target, data);
            }
        }, new Procedure<Runnable>() {
            @Override
            public void execute(Runnable result) {
                if (result != null)
                    result.run();
            }

            @Override
            public void exception(Throwable t) {
                ErrorLogger.defaultLogError(t);
            }
        });
    }

}
