/*******************************************************************************
 * Copyright (c) 2007, 2012 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.swt;

import java.util.List;

import org.eclipse.core.expressions.PropertyTester;
import org.eclipse.jface.viewers.ISelection;
import org.simantics.DatabaseJob;
import org.simantics.Simantics;
import org.simantics.browsing.ui.BuiltinKeys;
import org.simantics.browsing.ui.NodeContext;
import org.simantics.browsing.ui.common.node.IDeletable;
import org.simantics.browsing.ui.common.node.IModifiable;
import org.simantics.browsing.ui.common.node.IRefreshable;
import org.simantics.browsing.ui.model.queries.IsNodeContextModifiable;
import org.simantics.browsing.ui.model.queries.IsNodeContextRemovable;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.common.utils.RequestUtil;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.SelectionHints;
import org.simantics.ui.SimanticsUI;
import org.simantics.utils.ui.ISelectionUtils;

/**
 * A JFace property tester extension for the Eclipse command framework that is
 * meant for working with {@link NodeContext} instances of the Simantics
 * browsing framework.
 * 
 * <p>
 * This tester expects to receive IStructuredSelections that contain NodeContext
 * instances.
 * 
 * <p>
 * It supports testing of the following properties:
 * <ul>
 * <li>nodeClass - tests if the {@link NodeContext} input (see
 * {@link BuiltinKeys#INPUT}) is an instance of the class specified as an
 * argument. The class must be given as a qualified class name.</li>
 * <li>deletable - for testing whether a NodeContext can be considered
 * generically deletable, see <code>org.eclipse.ui.edit.delete</code> command.
 * No arguments required.</li>
 * <li>modifiable - for testing whether a NodeContext can be considered
 * generically modifiable (either in-line or not), see
 * <code>org.eclipse.ui.edit.rename</code> command. No arguments required.</li>
 * <li>refreshable - for testing whether a NodeContext can be considered
 * generically refreshable, see <code>org.eclipse.ui.file.refresh</code>
 * command. No arguments required.</li>
 * </ul>
 * 
 * @author Tuukka Lehtonen
 */
public class NodePropertyTester extends PropertyTester {

    /**
     * Tests if the received object within the model browser tree node is an
     * instance of the class specified as the argument.
     */
    private static final String NODE_CLASS = "nodeClass";

    /**
     * Tests if the received object is considered deletable.
     */
    private static final String DELETABLE = "deletable";

    /**
     * Tests if the received object is considered modifiable.
     */
    private static final String MODIFIABLE = "modifiable";

    /**
     * Tests if the received object is considered refreshable.
     */
    private static final String REFRESHABLE = "refreshable";

    ContextTester deletableTester = new DeletableTester();
    ContextTester modifiableTester = new ModifiableTester();
    ContextTester refreshableTester = new RefreshableTester();

    @Override
    public boolean test(final Object receiver, final String property, final Object[] args, final Object expectedValue) {
//        System.out.println("TEST: " + receiver + ", " + property + ", " + Arrays.toString(args) + ", " + expectedValue);
        if (!(receiver instanceof ISelection))
            return false;

        List<NodeContext> ncs = ISelectionUtils.getPossibleKeys((ISelection) receiver, SelectionHints.KEY_MAIN, NodeContext.class);
        
        if (ncs.isEmpty())
            return false;

        if (NODE_CLASS.equals(property)) {
            return testCollection(ncs, new NodeClassTester(args));
        } else if (DELETABLE.equals(property)) {
            return testCollection(ncs, deletableTester);
        } else if (MODIFIABLE.equals(property)) {
            if (ncs.size() == 1)
                return testCollection(ncs, modifiableTester);
            return false;
        } else if (REFRESHABLE.equals(property)) {
            return testCollection(ncs, refreshableTester);
        }

        return false;
    }

    static interface ContextTester {
        boolean test(NodeContext c);
    }

    public static class NodeClassTester implements ContextTester {
        Object[] args;
        NodeClassTester(Object[] args) {
            this.args = args;
        }
        @Override
        public boolean test(NodeContext nc) {
            try {
                Object input = nc.getConstant(BuiltinKeys.INPUT);
                Class<?> inputClass = input.getClass();
//                System.out.println("input: " + input);
                boolean assignable = false;
                for (Object o : args) {
                    Class<?> clazz = Class.forName((String) o, true, input.getClass().getClassLoader());
//                    System.out.println("  ARG " + o + " CLAZZ: " + clazz);
                    if (clazz.isAssignableFrom(inputClass)) {
//                        System.out.println("  assignable!");
                        assignable = true;
                        break;
                    }
                }
                return assignable;
            } catch (ClassNotFoundException e) {
            }
            return false;
        }
    }

    public static class DeletableTester implements ContextTester {
        @Override
        public boolean test(NodeContext nc) {
            Object input = nc.getConstant(BuiltinKeys.INPUT);
            if (input instanceof Resource) {
                Session session = Simantics.peekSession();
                try {
                    if (session != null && !DatabaseJob.inProgress())
                        return RequestUtil.trySyncRequest(
                                session,
                                SimanticsUI.UI_THREAD_REQUEST_START_TIMEOUT,
                                SimanticsUI.UI_THREAD_REQUEST_EXECUTION_TIMEOUT,
                                false,
                                new IsNodeContextRemovable( nc ) );
                } catch (DatabaseException | InterruptedException e) {
                }
                return false;
            } else if (input instanceof IDeletable) {
                // OK, this is something that should be considered
                // deletable, as long as we can find
                // a method for performing the deletion.
                return true;
            }
            return false;
        }
    }

    public static class ModifiableTester implements ContextTester {
        @Override
        public boolean test(NodeContext nc) {
            Object input = nc.getConstant(BuiltinKeys.INPUT);
            if (input instanceof Resource) {
                Session session = Simantics.peekSession();
                try {
                    if (session != null && !DatabaseJob.inProgress())
                        return RequestUtil.trySyncRequest(
                                session,
                                SimanticsUI.UI_THREAD_REQUEST_START_TIMEOUT,
                                SimanticsUI.UI_THREAD_REQUEST_EXECUTION_TIMEOUT,
                                false,
                                new IsNodeContextModifiable( nc ) );
                } catch (DatabaseException | InterruptedException e) {
                }
                return false;
            } else if (input instanceof IModifiable) {
                return true;
            }
            return false;
        }
    }

    public static class RefreshableTester implements ContextTester {
        @Override
        public boolean test(NodeContext nc) {
            return nc.getConstant(BuiltinKeys.INPUT) instanceof IRefreshable;
        }
    }

    private boolean testCollection(List<NodeContext> receiver, ContextTester tester) {
        if (receiver.isEmpty())
            return false;

        int fails = 0;
        for (NodeContext nc : receiver) {
            if (!tester.test(nc))
                ++fails;
        }
        return fails > 0 ? false : true;
    }

}
