/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.model.modifiers;

import org.simantics.browsing.ui.common.ErrorLogger;
import org.simantics.browsing.ui.content.Labeler.Modifier;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.RequestProcessor;
import org.simantics.db.Session;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.common.utils.Logger;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.variable.InputValidator;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.layer0.variable.VariableWrite;
import org.simantics.db.layer0.variable.Variables;
import org.simantics.scl.runtime.function.Function1;
import org.simantics.utils.datastructures.Callback;

/**
 * @author Tuukka Lehtonen
 */
public class VariableModifier2 implements Modifier {

    final protected Session     session;
    final protected Variable    variable;

    private String              initialValue;
    protected Throwable         modifierFailed;
    protected InputValidator    variableInputValidator;

    public VariableModifier2(RequestProcessor processor, Variable variable) {
        this.variable = variable;
        this.session = processor.getSession();
        initializeModifier(processor);
    }

    protected void initializeModifier(RequestProcessor processor) {
    	
        try {
        	
            processor.syncRequest(new ReadRequest() {
                @Override
                public void run(ReadGraph graph) throws DatabaseException {
                    initialValue = getInitialValue(graph);
                    initializeValidator(graph, variable);
                }
            });
            
        } catch (DatabaseException e) {
            Logger.defaultLogError("Modifier initialization failed, see exception for details.", e);
            modifierFailed = e;
        }
        
    }

    protected void initializeValidator(ReadGraph graph, Variable var) throws DatabaseException {
    	
//    	Resource validator = variable.getPossiblePropertyValue(graph, Variables.INPUT_VALIDATOR);
//    	if(validator != null) {
//    		variableInputValidator = graph.getPossibleContextualAdapter(validator, variable, Variable.class, InputValidator.class);
//    	}
    	
        final Function1<String, String> validator = variable.getPossiblePropertyValue(graph, Variables.INPUT_VALIDATOR);
        if (validator != null)
            variableInputValidator = new InputValidator() {
                
                @Override
                public String isValid(Object newValue) {
                    return validator.apply((String)newValue);
                }
            };
    	
        
    }

    protected void doModify(final String label) {
        session.asyncRequest(new VariableWrite(variable, label), new Callback<DatabaseException>() {
            @Override
            public void run(DatabaseException parameter) {
                if (parameter != null)
                    ErrorLogger.defaultLogError(parameter);
                else
                	modifySuccessful();
            }
        });
    }

    protected void modifySuccessful() {
	}

    protected String getInitialValue(ReadGraph graph) throws DatabaseException {
    	return variable.getValue(graph, Bindings.STRING);
    }

    @Override
    public String getValue() {
        return initialValue;
    }

    @Override
    public String isValid(String label) {
    	
        if (modifierFailed != null)
            return "Could not resolve validator for this value, modification denied. Reason: " + modifierFailed.getMessage();

        if (variableInputValidator != null)
            return variableInputValidator.isValid(label);

        return null;
        
    }

    @Override
    public final void modify(String label) {
        if (modifierFailed != null)
            // Should never end up here, isValid should prevent it.
            throw new Error("modifier failed: " + modifierFailed.getMessage());
        doModify(label);
    }

};
