/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.graph.impl;

import java.util.function.Consumer;

import org.simantics.browsing.ui.BuiltinKeys;
import org.simantics.browsing.ui.DataSource;
import org.simantics.browsing.ui.NodeContext;
import org.simantics.browsing.ui.PrimitiveQueryUpdater;
import org.simantics.browsing.ui.common.viewpoints.ViewpointStub;
import org.simantics.browsing.ui.content.Viewpoint;
import org.simantics.browsing.ui.graph.impl.request.ParametrizedResourceQuery;
import org.simantics.db.ReadGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.procedure.Listener;
import org.simantics.utils.datastructures.Pair;

/**
 * Implement {@link #children(ReadGraph)} and {@link #hasChildren(ReadGraph)}.
 * 
 * @author Tuukka Lehtonen
 */
public abstract class LazyParametrizedViewpoint extends ViewpointStub {

	/**
	 * Needed for separating childQuery and hasChildQuery from each other in the
	 * equals sense.
	 */
	private static final Object                              CHILDREN     = new Object();
	private static final Object                              HAS_CHILDREN = new Object();

	private final ParametrizedResourceQuery<NodeContext[]> childQuery;
	private final ParametrizedResourceQuery<Boolean>        hasChildQuery;

    private final Listener<NodeContext[]>              childQueryProcedure;
    private final Listener<Boolean>                     hasChildQueryProcedure;

    private final PrimitiveQueryUpdater                      updater;
    private final NodeContext                               context;
    private final BuiltinKeys.ViewpointKey                   key;

	/**
	 * @return
	 */
	public abstract NodeContext[] children(ReadGraph graph) throws DatabaseException;

	/**
	 * @param graph
	 * @return
	 */
	public abstract Boolean hasChildren(ReadGraph graph) throws DatabaseException;


	public LazyParametrizedViewpoint(PrimitiveQueryUpdater updater, NodeContext context, BuiltinKeys.ViewpointKey key, Object... parameters) {
		assert updater != null;
		assert context != null;
		assert key != null;

		this.updater = updater;
		this.context = context;
		this.key = key;

		this.childQuery = new ParametrizedResourceQuery<NodeContext[]>(Pair.make(getClass(), CHILDREN), context, parameters) {

			@Override
			public NodeContext[] perform(ReadGraph graph) throws DatabaseException {
				return  children(graph);
			}

		};

		this.childQueryProcedure = new Listener<NodeContext[]>() {

			@Override
			public void execute(NodeContext[] result) {
				replaceChildrenResult(result);
			}

			@Override
			public boolean isDisposed() {
				return LazyParametrizedViewpoint.this.updater.isDisposed();
			}

			public void exception(Throwable t) {
				System.out.print("LazyParametrizedViewpoint2.childQuery failed: ");
				t.printStackTrace();
			}

		};

		this.hasChildQuery = new ParametrizedResourceQuery<Boolean>(Pair.make(getClass(), HAS_CHILDREN), context, parameters) {

			@Override
			public Boolean perform(ReadGraph graph) throws DatabaseException {
				return hasChildren(graph);
			}

		};

		this.hasChildQueryProcedure = new Listener<Boolean>() {

			@Override
			public void execute(Boolean result) {
				replaceHasChildrenResult(result);
			}

			@Override
			public boolean isDisposed() {
				return LazyParametrizedViewpoint.this.updater.isDisposed();
			}

			public void exception(Throwable t) {
				System.out.print("LazyParametrizedViewpoint2.hasChildQuery failed: ");
				t.printStackTrace();
			}

		};

	}

	public NodeContext getContext() {
		return context;
	}

	@Override
	public NodeContext[] getChildren() {
		if (children == Viewpoint.PENDING_CHILDREN) {
			DataSource<ReadGraph> source = updater.getDataSource(ReadGraph.class);
			if (source != null) {
				source.schedule(graph -> graph.asyncRequest(childQuery, childQueryProcedure));
			}
		}

		return children;
	}

	@Override
	public Boolean getHasChildren() {
		if (hasChildren == Viewpoint.PENDING_HAS_CHILDREN) {
			DataSource<ReadGraph> source = updater.getDataSource(ReadGraph.class);
			if (source != null) {
				source.schedule(new Consumer<ReadGraph>() {
					@Override
					public void accept(ReadGraph source) {
						source.asyncRequest(hasChildQuery, hasChildQueryProcedure);
					}
				});
			}
		}

		return hasChildren;
	}

	protected void replaceChildrenResult(NodeContext[] result) {
		setChildren(updater, result);
		updater.scheduleReplace(context, key, this);
	}

	protected void replaceHasChildrenResult(Boolean result) {
		setHasChildren(result);
		updater.scheduleReplace(context, key, this);
	}

	/**
	 * @param <T>
	 * @param clazz
	 * @return input of the specified class
	 * @throws ClassCastException if the input class does not match the
	 *         specified class
	 * @throws NullPointerException if the input is null
	 */
	 protected <T> T getInput(Class<T> clazz) throws ClassCastException {
		 Object o = context.getConstant(BuiltinKeys.INPUT);
		 if (o == null)
			 throw new NullPointerException("null input");
		 return clazz.cast(o);
	 }

	 /**
	  * @param <T>
	  * @param clazz
	  * @return <code>null</code> if input is <code>null</code> or if the class does not match
	  */
	 protected <T> T tryGetInput(Class<T> clazz) {
		 Object o = context.getConstant(BuiltinKeys.INPUT);
		 if (o != null && clazz.isInstance(o))
			 return clazz.cast(o);
		 return null;
	 }

}
