/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.graph.impl;

import org.simantics.browsing.ui.content.Labeler.Modifier;
import org.simantics.browsing.ui.graph.impl.request.GetLabel;
import org.simantics.databoard.Bindings;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.util.Layer0Utils;
import org.simantics.db.request.Write;
import org.simantics.layer0.Layer0;
import org.simantics.utils.datastructures.Callback;
import org.simantics.utils.ui.ErrorLogger;
import org.simantics.utils.ui.ExceptionUtils;

/**
 * NOTE: By default writing Label property, NOT Name property! Name is
 * considered a unique immutable identifier for the model. This helps with
 * writing stuff on disk and keeping the names coherent.
 * 
 * @author Tuukka Lehtonen
 */
public class LabelModifier implements Modifier, Callback<DatabaseException> {

    private final Session  session;
    protected final Resource resource;
    protected final Resource targetProperty;

    /**
     * @param session database access point
     * @param resource the resource for which to modify the label
     * @param writeCallback a callback that is invoked once the graph write has
     *        finished. The exception will be null if the write was successful.
     */
    public LabelModifier(Session session, Resource resource) {
        this(session, resource, session.getService(Layer0.class).HasLabel);
    }

    /**
     * @param session database access point
     * @param resource the resource for which to modify the label
     * @param targetProperty the relation of the target property to be claimed
     *        by the modifier.
     * @param writeCallback a callback that is invoked once the graph write has
     *        finished. The exception will be null if the write was successful.
     */
    public LabelModifier(Session session, Resource resource, Resource targetProperty) {
        this.session = session;
        this.resource = resource;
        this.targetProperty = targetProperty;
    }

    @Override
    public String getValue() {
        try {
            return session.syncRequest(new GetLabel(resource));
        } catch (DatabaseException e) {
            ErrorLogger.defaultLogWarning("Problem reading current label, see exception for details.", e);
            return "";
        }
    }

    @Override
    public String isValid(String label) {
        if (label.isEmpty())
            return "Empty label not allowed";
        return null;
    }

    @Override
    public void modify(String label) {
        session.asyncRequest(getWriteRequest(label), this);
    }

    protected Write getWriteRequest(final String label) {
        return new WriteRequest() {
            @Override
            public void perform(WriteGraph g) throws DatabaseException {
                g.markUndoPoint();
                Layer0Utils.addCommentMetadata(g, "Renamed " + g.getRelatedValue2(resource, Layer0.getInstance(g).HasName, Bindings.STRING) + " to " + label + " " + resource.toString());
                g.claimLiteral(resource, targetProperty, label);
            }
        };
    }

    @Override
    public void run(DatabaseException parameter) {
        if (parameter != null) {
            ExceptionUtils.logError("Label modification failed, see exception for details.", parameter);
        }
    }

}
