/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.graph.impl;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.eclipse.jface.resource.ResourceManager;
import org.simantics.browsing.ui.BuiltinKeys.CheckedStateKey;
import org.simantics.browsing.ui.BuiltinKeys.ImageDecoratorKey;
import org.simantics.browsing.ui.BuiltinKeys.ImagerKey;
import org.simantics.browsing.ui.BuiltinKeys.LabelDecoratorKey;
import org.simantics.browsing.ui.BuiltinKeys.LabelerKey;
import org.simantics.browsing.ui.BuiltinKeys.ViewpointKey;
import org.simantics.browsing.ui.CheckedState;
import org.simantics.browsing.ui.NodeContext;
import org.simantics.browsing.ui.PrimitiveQueryUpdater;
import org.simantics.browsing.ui.Tester;
import org.simantics.browsing.ui.common.EvaluatorData;
import org.simantics.browsing.ui.common.EvaluatorData.Evaluator;
import org.simantics.browsing.ui.common.EvaluatorData.EvaluatorTree;
import org.simantics.browsing.ui.common.EvaluatorDataImpl;
import org.simantics.browsing.ui.common.EvaluatorImpl;
import org.simantics.browsing.ui.common.exception.InvalidBrowserIdException;
import org.simantics.browsing.ui.common.extension.CheckedStateContributorBindingExtensionManager;
import org.simantics.browsing.ui.common.extension.ComparableContextContributorBindingExtensionManager;
import org.simantics.browsing.ui.common.extension.EvaluatorBindingExtensionManager;
import org.simantics.browsing.ui.common.extension.EvaluatorFactory;
import org.simantics.browsing.ui.common.extension.ImageDecoratorContributorBindingExtensionManager;
import org.simantics.browsing.ui.common.extension.ImagerContributorBindingExtensionManager;
import org.simantics.browsing.ui.common.extension.LabelDecoratorContributorBindingExtensionManager;
import org.simantics.browsing.ui.common.extension.LabelerContributorBindingExtensionManager;
import org.simantics.browsing.ui.common.extension.ViewpointContributionContributorBindingExtensionManager;
import org.simantics.browsing.ui.content.CheckedStateFactory;
import org.simantics.browsing.ui.content.ComparableContextFactory;
import org.simantics.browsing.ui.content.ContributorBinding;
import org.simantics.browsing.ui.content.ImageDecorator;
import org.simantics.browsing.ui.content.ImageDecoratorFactory;
import org.simantics.browsing.ui.content.Imager;
import org.simantics.browsing.ui.content.ImagerFactory;
import org.simantics.browsing.ui.content.LabelDecorator;
import org.simantics.browsing.ui.content.LabelDecoratorFactory;
import org.simantics.browsing.ui.content.Labeler;
import org.simantics.browsing.ui.content.LabelerFactory;
import org.simantics.browsing.ui.content.Viewpoint;
import org.simantics.browsing.ui.content.ViewpointContributionFactory;
import org.simantics.browsing.ui.content.ViewpointFactory;
import org.simantics.browsing.ui.model.InvalidContribution;
import org.simantics.browsing.ui.model.browsecontexts.BrowseContext;
import org.simantics.db.ReadGraph;
import org.simantics.db.RequestProcessor;
import org.simantics.db.Resource;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.ResourceNotFoundException;

/**
 * An entry point for loading an EvaluatorData in a data-oriented fashion. Also
 * adds contributions made by extensions of the related extension points.
 * 
 * @author Antti Villberg
 */
public class Evaluators {

    public static final boolean DEBUG = false;

    public static void create(EvaluatorData data, final ResourceManager resourceManager,
            final Map<String, Collection<ContributorBinding<ViewpointContributionFactory>>> viewpointContributions,
            final Set<ContributorBinding<ComparableContextFactory>> comparableContextContributions,
            final Set<ContributorBinding<LabelerFactory>> labelerContributions,
            final Set<ContributorBinding<CheckedStateFactory>> checkStateContributions,
            final Set<ContributorBinding<LabelDecoratorFactory>> labelDecoratorContributions,
            final Set<ContributorBinding<ImagerFactory>> imagerContributions,
            final Set<ContributorBinding<ImageDecoratorFactory>> imageDecoratorContributions) {

        Map<Class<?>, Evaluator> evaluators = new HashMap<Class<?>, Evaluator>();

        HashMap<String, ContributionViewpointFactory> viewpointFactories = new HashMap<String, ContributionViewpointFactory>();

        for(final Map.Entry<String, Collection<ContributorBinding<ViewpointContributionFactory>>> entry : viewpointContributions.entrySet()) {
            Collection<ViewpointContributionFactory> factories = new ArrayList<ViewpointContributionFactory>();
            for(ContributorBinding<ViewpointContributionFactory> binding : entry.getValue()) factories.add(binding.getContributor().getFactory());
            viewpointFactories.put(entry.getKey(), new ContributionViewpointFactory(entry.getKey(), factories));
        }

        for(final Map.Entry<String, Collection<ContributorBinding<ViewpointContributionFactory>>> entry : viewpointContributions.entrySet()) {
            ContributionViewpointFactory factory = viewpointFactories.get(entry.getKey());
            for(ContributorBinding<ViewpointContributionFactory> binding : entry.getValue()) {
                Tester test = binding.getContributor().getNodeContextTester();
                Class<?> clazz = binding.getContributor().getInputClass();
                Evaluator evaluator = evaluators.get(clazz);
                if(evaluator == null) {
                    evaluator = data.newEvaluator();
                    evaluators.put(clazz, evaluator);
                }
                if(test == null) {
                    if (DEBUG)
                        System.out.println("add root viewpoint " + factory + " to class " + clazz.getSimpleName());
                    evaluator.addViewpoint(factory, binding.getPreference());
                } else {
                    if (DEBUG)
                        System.out.println("add viewpoint " + factory + " to class " + clazz.getSimpleName() + " with tester " + test);
                    EvaluatorTree<ViewpointFactory> branch = ((EvaluatorImpl)evaluator).getViewpointTree().addBranch(test);
                    branch.addFactory(factory, binding.getPreference());
                }
            }
        }

        for(final ContributorBinding<ComparableContextFactory> binding : comparableContextContributions) {

            ComparableContextFactory factory = binding.getContributor().getFactory();
            Tester test = binding.getContributor().getNodeContextTester();
            Class<?> clazz = binding.getContributor().getInputClass();
            Evaluator evaluator = evaluators.get(clazz);
            if(evaluator == null) {
                evaluator = data.newEvaluator();
                evaluators.put(clazz, evaluator);
            }
            if(test == null) {
                if (DEBUG)
                    System.out.println("add root comparable context " + factory + " to class " + clazz.getSimpleName());
                evaluator.addComparableContext(factory, binding.getPreference());
            } else {
                if (DEBUG)
                    System.out.println("add comparable context " + factory + " to class " + clazz.getSimpleName() + " with tester " + test);
                EvaluatorTree<ComparableContextFactory> branch = ((EvaluatorImpl)evaluator).getComparableContextTree().addBranch(test);
                branch.addFactory(factory, binding.getPreference());
            }
        }

        for(final ContributorBinding<LabelerFactory> binding : labelerContributions) {

            LabelerFactory factory = binding.getContributor().getFactory();
            Tester test = binding.getContributor().getNodeContextTester();
            Class<?> clazz = binding.getContributor().getInputClass();
            Evaluator evaluator = evaluators.get(clazz);
            if(evaluator == null) {
                evaluator = data.newEvaluator();
                evaluators.put(clazz, evaluator);
            }
            if(test == null) {
                if (DEBUG)
                    System.out.println("add root labeler " + factory + " to class " + clazz.getSimpleName());
                evaluator.addLabeler(factory, binding.getPreference());
            } else {
                if (DEBUG)
                    System.out.println("add labeler " + factory + " to class " + clazz.getSimpleName() + " with tester " + test);
                EvaluatorTree<LabelerFactory> branch = ((EvaluatorImpl)evaluator).getLabelerTree().addBranch(test);
                branch.addFactory(factory, binding.getPreference());
            }
        }

        for(final ContributorBinding<CheckedStateFactory> binding : checkStateContributions) {

            CheckedStateFactory factory = binding.getContributor().getFactory();
            Tester test = binding.getContributor().getNodeContextTester();
            Class<?> clazz = binding.getContributor().getInputClass();
            Evaluator evaluator = evaluators.get(clazz);
            if(evaluator == null) {
                evaluator = data.newEvaluator();
                evaluators.put(clazz, evaluator);
            }
            if(test == null) {
                if (DEBUG)
                    System.out.println("add root check state " + factory + " to class " + clazz.getSimpleName());
                evaluator.addCheckState(factory, binding.getPreference());
            } else {
                if (DEBUG)
                    System.out.println("add check state " + factory + " to class " + clazz.getSimpleName() + " with tester " + test);
                EvaluatorTree<CheckedStateFactory> branch = ((EvaluatorImpl)evaluator).getCheckStateTree().addBranch(test);
                branch.addFactory(factory, binding.getPreference());
            }
        }

        for(final ContributorBinding<LabelDecoratorFactory> binding : labelDecoratorContributions) {

            LabelDecoratorFactory factory = binding.getContributor().getFactory();
            Tester test = binding.getContributor().getNodeContextTester();
            Class<?> clazz = binding.getContributor().getInputClass();
            Evaluator evaluator = evaluators.get(clazz);
            if(evaluator == null) {
                evaluator = data.newEvaluator();
                evaluators.put(clazz, evaluator);
            }
            if(test == null) {
                if (DEBUG)
                    System.out.println("add root label decorator " + factory + " to class " + clazz.getSimpleName());
                evaluator.addLabelDecorator(factory, binding.getPreference());
            } else {
                if (DEBUG)
                    System.out.println("add label decorator " + factory + " to class " + clazz.getSimpleName() + " with tester " + test);
                EvaluatorTree<LabelDecoratorFactory> branch = ((EvaluatorImpl)evaluator).getLabelDecoratorTree().addBranch(test);
                branch.addFactory(factory, binding.getPreference());
            }
        }

        for(final ContributorBinding<ImagerFactory> binding : imagerContributions) {

            ImagerFactory factory = binding.getContributor().getFactory();
            Tester test = binding.getContributor().getNodeContextTester();
            Class<?> clazz = binding.getContributor().getInputClass();
            Evaluator evaluator = evaluators.get(clazz);
            if(evaluator == null) {
                evaluator = data.newEvaluator();
                evaluators.put(clazz, evaluator);
            }
            if(test == null) {
                if (DEBUG)
                    System.out.println("add root imager " + factory + " to class " + clazz.getSimpleName());
                evaluator.addImager(factory, binding.getPreference());
            } else {
                if (DEBUG)
                    System.out.println("add imager " + factory + " to class " + clazz.getSimpleName() + " with tester " + test);
                EvaluatorTree<ImagerFactory> branch = ((EvaluatorImpl)evaluator).getImagerTree().addBranch(test);
                branch.addFactory(factory, binding.getPreference());
            }
        }

        for(final ContributorBinding<ImageDecoratorFactory> binding : imageDecoratorContributions) {

            ImageDecoratorFactory factory = binding.getContributor().getFactory();
            Tester test = binding.getContributor().getNodeContextTester();
            Class<?> clazz = binding.getContributor().getInputClass();
            Evaluator evaluator = evaluators.get(clazz);
            if(evaluator == null) {
                evaluator = data.newEvaluator();
                evaluators.put(clazz, evaluator);
            }
            if(test == null) {
                if (DEBUG)
                    System.out.println("add root image decorator " + factory + " to class " + clazz.getSimpleName());
                evaluator.addImageDecorator(factory, binding.getPreference());
            } else {
                if (DEBUG)
                    System.out.println("add image decorator " + factory + " to class " + clazz.getSimpleName() + " with tester " + test);
                EvaluatorTree<ImageDecoratorFactory> branch = ((EvaluatorImpl)evaluator).getImageDecoratorTree().addBranch(test);
                branch.addFactory(factory, binding.getPreference());
            }
        }

        for(Map.Entry<Class<?>, Evaluator> entry : evaluators.entrySet()) {
            data.addEvaluator(entry.getKey(), entry.getValue());
        }

    }

    /**
     * @param processor
     * @param browseContexts
     * @param resourceManager
     * @param data
     * @return the URIs of the actual set of browse contexts loaded by this
     *         method. If an included browse context resource has no URI, it
     *         will not be included in this set.
     */
    public static Set<String> loadModelled(
            RequestProcessor processor, 
            final Set<String> browseContexts, 
            final ResourceManager resourceManager,
            final EvaluatorDataImpl data,
            final boolean useNodeBrowseContexts,
            final boolean useNodeActionContexts) {

        final Set<String> allBrowseContexts = new HashSet<String>(browseContexts);

        // Create evaluator
        final EvaluatorImpl eval = new EvaluatorImpl();
        data.addEvaluator(Object.class, eval);  
        
        try {

            // Load data for evaluator
            processor.syncRequest(new ReadRequest() {
                
                @Override
                public void run(ReadGraph graph) throws DatabaseException {

                    // Map browse context names to resources
                    Collection<Resource> browseContextResources = new ArrayList<Resource>(browseContexts.size());
                    for(String browseContext : browseContexts) {
                        try {
                            browseContextResources.add(graph.getResource(browseContext));
                        } catch(ResourceNotFoundException e) {
                            // Expected result, if no modelled contributions exist.
                            //System.err.println("Didn't find " + browseContext + " while loading model browser.");
                        } catch(DatabaseException e) {
                            System.err.println("Didn't find " + browseContext + " while loading model browser.");
                            e.printStackTrace();
                        }
                    }

                    // Load browse context
                    try {
                        final BrowseContext browseContext = 
                            BrowseContext.create(graph, browseContextResources);
                        
                        data.setBrowseContext(browseContext);

                        // Get URI's for all found browse contexts to fill return value
                        for (Resource context : BrowseContext.findSubcontexts(graph, browseContextResources)) {
                            String uri = graph.getPossibleURI(context);
                            if (uri != null)
                                allBrowseContexts.add(uri);
                        }

                        // Fill evaluator
                        eval.addLabeler(new LabelerFactory() {
                            @Override
                            public Labeler create(PrimitiveQueryUpdater updater, final NodeContext context,
                                    LabelerKey key) {
                                return new EvaluatorLabeler(updater, context, key, browseContext, useNodeBrowseContexts);
                            }
                        }, 0.0);

                        eval.addImager(new ImagerFactory() {
                            @Override
                            public Imager create(PrimitiveQueryUpdater updater, NodeContext context,
                                    ImagerKey key) {
                                return new EvaluatorImager(updater, context, key, browseContext, useNodeBrowseContexts);
                            }
                        }, 0.0);

                        eval.addCheckState(new CheckedStateFactory() {

                            @Override
                            public CheckedState create(PrimitiveQueryUpdater updater,
                                    NodeContext context, CheckedStateKey key) {
                                return new EvaluatorCheckedState(updater, context, key, browseContext, useNodeBrowseContexts).getState();
                            }
                        }, 0.0);

                        eval.addLabelDecorator(new LabelDecoratorFactory() {
                            @Override
                            public LabelDecorator create(PrimitiveQueryUpdater updater,
                                    NodeContext context, LabelDecoratorKey key) {
                                return new EvaluatorLabelDecorator(updater, context, key, browseContext, useNodeBrowseContexts);
                            }
                        }, 0.0);

                        eval.addImageDecorator(new ImageDecoratorFactory() {
                            @Override
                            public ImageDecorator create(PrimitiveQueryUpdater updater,
                                    NodeContext context, ImageDecoratorKey key) {
                                return new EvaluatorImageDecorator(updater, context, key, browseContext, useNodeBrowseContexts);
                            }
                        }, 0.0);

                        eval.addViewpoint(new ViewpointFactory() {
                            @Override
                            public Viewpoint create(PrimitiveQueryUpdater updater, 
                                    final NodeContext context,
                                    ViewpointKey key) {
                                return new EvaluatorViewpoint(updater, context, key, browseContext, useNodeBrowseContexts, useNodeActionContexts);
                            }
                        }, 0.0);
                    } catch (InvalidContribution e) {
                        e.printStackTrace();
                        return;
                    }

                }

            });

        } catch (DatabaseException e) {

            e.printStackTrace();

        }

        return allBrowseContexts;
    }

    public static void loadExtensions(RequestProcessor processor, final Set<String> browseContexts, final ResourceManager resourceManager,
            final Map<String, Collection<ContributorBinding<ViewpointContributionFactory>>> viewpointContributions,
            final Set<ContributorBinding<ComparableContextFactory>> comparableContextContributions,
            final Set<ContributorBinding<LabelerFactory>> labelerContributions,
            final Set<ContributorBinding<CheckedStateFactory>> checkStateContributions,
            final Set<ContributorBinding<LabelDecoratorFactory>> labelDecoratorContributions,
            final Set<ContributorBinding<ImagerFactory>> imagerContributions,
            final Set<ContributorBinding<ImageDecoratorFactory>> imageDecoratorContributions) {

        for(ContributorBinding<ViewpointContributionFactory> binding : ViewpointContributionContributorBindingExtensionManager.getInstance().getBoundContributions(browseContexts)) {
            String viewpointId = binding.getContributor().getFactory().getViewpointId();
            Collection<ContributorBinding<ViewpointContributionFactory>> viewpoints = viewpointContributions.get(viewpointId);
            if(viewpoints == null) {
                viewpoints = new ArrayList<ContributorBinding<ViewpointContributionFactory>>();
                viewpointContributions.put(viewpointId, viewpoints);
            }
            viewpoints.add(binding);
        }

        for(final ContributorBinding<ComparableContextFactory> binding : ComparableContextContributorBindingExtensionManager.getInstance().getBoundContributions(browseContexts)) {
            comparableContextContributions.add(binding);
        }

        for(final ContributorBinding<LabelerFactory> binding : LabelerContributorBindingExtensionManager.getInstance().getBoundContributions(browseContexts)) {
            labelerContributions.add(binding);
        }

        for(final ContributorBinding<CheckedStateFactory> binding : CheckedStateContributorBindingExtensionManager.getInstance().getBoundContributions(browseContexts)) {
            checkStateContributions.add(binding);
        }

        for(final ContributorBinding<LabelDecoratorFactory> binding : LabelDecoratorContributorBindingExtensionManager.getInstance().getBoundContributions(browseContexts)) {
            labelDecoratorContributions.add(binding);
        }

        for(final ContributorBinding<ImagerFactory> binding : ImagerContributorBindingExtensionManager.getInstance().getBoundContributions(browseContexts)) {
            imagerContributions.add(binding);
        }

        for(final ContributorBinding<ImageDecoratorFactory> binding : ImageDecoratorContributorBindingExtensionManager.getInstance().getBoundContributions(browseContexts)) {
            imageDecoratorContributions.add(binding);
        }

    }

    public static EvaluatorData load(RequestProcessor processor, Set<String> browseContexts, ResourceManager resourceManager) throws InvalidBrowserIdException {
        return load(processor, browseContexts, resourceManager, false, false);
    }

    public static EvaluatorData load(RequestProcessor processor, Set<String> browseContexts, ResourceManager resourceManager, boolean useNodeBrowseContexts, boolean useNodeActionContexts) throws InvalidBrowserIdException {

        Map<String, Collection<ContributorBinding<ViewpointContributionFactory>>> viewpointContributions = new HashMap<String, Collection<ContributorBinding<ViewpointContributionFactory>>>();
        Set<ContributorBinding<ComparableContextFactory>> comparableContextContributions = new HashSet<ContributorBinding<ComparableContextFactory>>();
        Set<ContributorBinding<LabelerFactory>> labelerContributions = new HashSet<ContributorBinding<LabelerFactory>>();
        Set<ContributorBinding<CheckedStateFactory>> checkStateContributions = new HashSet<ContributorBinding<CheckedStateFactory>>();
        Set<ContributorBinding<LabelDecoratorFactory>> labelDecoratorContributions = new HashSet<ContributorBinding<LabelDecoratorFactory>>();
        Set<ContributorBinding<ImagerFactory>> imagerContributions = new HashSet<ContributorBinding<ImagerFactory>>();
        Set<ContributorBinding<ImageDecoratorFactory>> imageDecoratorContributions = new HashSet<ContributorBinding<ImageDecoratorFactory>>();

        EvaluatorDataImpl data = new EvaluatorDataImpl();

        // Load modelled contributions
        Set<String> allBrowseContexts = loadModelled(processor, browseContexts, resourceManager, data, useNodeBrowseContexts, useNodeActionContexts);

        // Load extension point contributions
        loadExtensions(processor, allBrowseContexts, resourceManager, viewpointContributions, comparableContextContributions, labelerContributions, checkStateContributions, labelDecoratorContributions, imagerContributions, imageDecoratorContributions);

        // Create contributed evaluators
        create(data, resourceManager, viewpointContributions, comparableContextContributions, labelerContributions, checkStateContributions, labelDecoratorContributions, imagerContributions, imageDecoratorContributions);

        // Create plugin evaluators for the contexts
        for(EvaluatorFactory factory : EvaluatorBindingExtensionManager.getInstance().getBoundFactories(allBrowseContexts)) {
            data.addEvaluator(factory.getClazz(), factory.create(allBrowseContexts));
        }

        return data;

    }
    
}
