/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.graph.impl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.simantics.browsing.ui.content.Labeler.EnumerationModifier;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.RequestProcessor;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.UndoContext;
import org.simantics.db.VirtualGraph;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.ObjectsWithType;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.layer0.variable.Variables;
import org.simantics.db.request.Read;
import org.simantics.layer0.Layer0;
import org.simantics.utils.datastructures.Callback;
import org.simantics.utils.ui.ErrorLogger;

/**
 * @author Tuukka Lehtonen
 */
public class EnumerationVariableModifier2 implements EnumerationModifier {

    protected final Session      session;
    protected final UndoContext  undoContext;
    protected final Variable     variable;
    protected final Set<String> allowedValues;
    protected final String       defaultValue;

    protected Throwable        modifierFailed;

    public EnumerationVariableModifier2(RequestProcessor processor, UndoContext undoContext, Variable variable) {

    	this.session = processor.getSession();
        this.undoContext = undoContext;
        this.variable = variable;
        this.allowedValues = computeAllowedValues(processor, variable);
        this.defaultValue = computeDefaultValue(processor, variable);
        
    }

    protected String computeDefaultValue(RequestProcessor processor, final Variable variable) {
        try {
           return processor.syncRequest(new Read<String>() {
                @Override
                public String perform(ReadGraph graph) throws DatabaseException {
                	return variable.getPossiblePropertyValue(graph, Variables.LABEL);
                }
            });
        } catch (DatabaseException e) {
            return "";
        }
    }
    
    protected Set<String> computeAllowedValues(RequestProcessor processor, final Variable variable) {
        try {
           return processor.syncRequest(new Read<Set<String>>() {
                @Override
                public Set<String> perform(ReadGraph graph) throws DatabaseException {

                	Layer0 L0 = Layer0.getInstance(graph);
                	Set<String> result = new HashSet<String>();
            		Resource literal = variable.getPossibleRepresents(graph);
            		Resource type = graph.getSingleObject(literal, L0.PartOf);
            		for(Resource lit : graph.syncRequest(new ObjectsWithType(type, L0.ConsistsOf, type))) {
            			String label = graph.getPossibleRelatedValue(lit, L0.HasLabel);
            			if(label == null) label = graph.getRelatedValue(lit, L0.HasName);
            			result.add(label);
            		}
            		return result;
                	
                }
            });
        } catch (DatabaseException e) {
            return Collections.emptySet();
        }
    }

    public class Write extends WriteRequest {

        final private Variable variable;
        final private String label;

        public Write(Variable variable, String label) {
            super((VirtualGraph)null);
            this.variable = variable;
            this.label = label;
        }

        @Override
        public void perform(WriteGraph graph) throws DatabaseException {
        	variable.setValue(graph, label, Bindings.STRING);
        }

    }
    
    protected void doModify(final String label) {
        session.asyncRequest(new Write(variable, label),
                new Callback<DatabaseException>() {
            @Override
            public void run(DatabaseException parameter) {
                if (parameter != null)
                    ErrorLogger.defaultLogError(parameter);
            }
        });
    }

    @Override
    public String getValue() {
        return defaultValue;
    }

    @Override
    public String isValid(String label) {
        if (modifierFailed != null)
            return "Could not resolve validator for this value, modification denied. Reason: "
            + modifierFailed.getMessage();
        // Validity should already be enforced by the editing UI for
        // enumerations.
        return null;
    }

    @Override
    public final void modify(String label) {
        if (modifierFailed != null)
            // Should never end up here, isValid should prevent it.
            throw new Error("modifier failed: " + modifierFailed.getMessage());
        doModify(label);
    }

    @Override
    public List<String> getValues() {
    	ArrayList<String> result = new ArrayList<String>();
    	result.addAll(allowedValues);
    	return result;
    }

};
