/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.graph.impl;

import java.util.List;

import org.simantics.browsing.ui.content.Labeler.EnumerationModifier;
import org.simantics.db.ReadGraph;
import org.simantics.db.RequestProcessor;
import org.simantics.db.Session;
import org.simantics.db.UndoContext;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.layer0.variable.VariableWrite;
import org.simantics.utils.datastructures.Callback;
import org.simantics.utils.ui.ErrorLogger;

/**
 * @author Tuukka Lehtonen
 */
public class EnumerationVariableModifier implements EnumerationModifier {

    protected final Session      session;
    protected final UndoContext  undoContext;
    protected final Variable     variable;
    protected final List<String> allowedValues;
    protected final String       defaultValue;

    private String             initialValue;
    protected Throwable        modifierFailed;

    public EnumerationVariableModifier(RequestProcessor processor, UndoContext undoContext, Variable variable,
            List<String> allowedValues, String defaultValue) {
        this.session = processor.getSession();
        this.undoContext = undoContext;
        this.variable = variable;
        this.allowedValues = allowedValues;
        this.defaultValue = defaultValue;

        initializeModifier(processor);
    }

    protected void initializeModifier(RequestProcessor processor) {
        try {
            processor.syncRequest(new ReadRequest() {
                @Override
                public void run(ReadGraph graph) throws DatabaseException {
                    initialValue = getInitialValue(graph);
                }
            });
        } catch (DatabaseException e) {
            modifierFailed = e;
        }
    }

    protected void doModify(final String label) {
        session.asyncRequest(new VariableWrite(variable, label, null, null),
                new Callback<DatabaseException>() {
            @Override
            public void run(DatabaseException parameter) {
                if (parameter != null)
                    ErrorLogger.defaultLogError(parameter);
            }
        });
    }

    protected String getInitialValue(ReadGraph graph) throws DatabaseException {
        Object value = variable.getValue(graph);
        return value == null ? defaultValue : value.toString();
    }

    @Override
    public String getValue() {
        return initialValue;
    }

    @Override
    public String isValid(String label) {
        if (modifierFailed != null)
            return "Could not resolve validator for this value, modification denied. Reason: "
            + modifierFailed.getMessage();
        // Validity should already be enforced by the editing UI for
        // enumerations.
        return null;
    }

    @Override
    public final void modify(String label) {
        if (modifierFailed != null)
            // Should never end up here, isValid should prevent it.
            throw new Error("modifier failed: " + modifierFailed.getMessage());
        doModify(label);
    }

    @Override
    public List<String> getValues() {
        return allowedValues;
    }

};
