package org.simantics.scl.ui.issues;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.resource.ImageRegistry;
import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.ColumnViewerToolTipSupport;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ControlAdapter;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.ScrollBar;
import org.eclipse.swt.widgets.Table;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.IViewSite;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.part.ViewPart;
import org.simantics.scl.compiler.errors.ErrorSeverity;
import org.simantics.scl.osgi.SCLOsgi;
import org.simantics.scl.ui.Activator;
import org.simantics.scl.ui.editor2.OpenSCLDefinition;

public class SCLIssuesView extends ViewPart {

    private static final int COLUMN1_DEFAULT_WIDTH = 300;
    private static final int COLUMN2_MIN_WIDTH = 100;
    
    Composite parent;
    TableViewer tableViewer;
    TableViewerColumn column1;
    TableViewerColumn column2;

    ImageRegistry imageRegistry;
    
    SCLIssuesContentProvider issuesContentProvider = new SCLIssuesContentProvider();
    
    public SCLIssuesView() {
        super();
        imageRegistry = Activator.getInstance().getImageRegistry();
    }
    
    @Override
    public void init(IViewSite site, IMemento memento)
            throws PartInitException {
        super.init(site, memento);
        IAction action = new Action("Refresh") {
            @Override
            public void run() {
                SCLOsgi.MODULE_REPOSITORY.getSourceRepository().checkUpdates();
            }
        };
        action.setImageDescriptor(imageRegistry.getDescriptor("arrow_refresh"));
        site.getActionBars().getToolBarManager().add(action);
    }
    
    @Override
    public void createPartControl(Composite parent) {
        this.parent = parent;
        parent.setLayout(new FillLayout());
        tableViewer = new TableViewer(parent,
                SWT.FULL_SELECTION | SWT.SINGLE | SWT.V_SCROLL | SWT.H_SCROLL);
        ColumnViewerToolTipSupport.enableFor(tableViewer);
        tableViewer.setContentProvider(issuesContentProvider);

        Table table = tableViewer.getTable();
        table.setHeaderVisible(true);
        table.setLinesVisible(true);

        column1 = new TableViewerColumn(tableViewer, SWT.NONE);
        column1.getColumn().setText("Module");
        column1.getColumn().setWidth(COLUMN1_DEFAULT_WIDTH);
        column1.getColumn().setResizable(true);
        column1.setLabelProvider(new ColumnLabelProvider() {
            @Override
            public String getText(Object element) {
                SCLIssuesTableEntry entry = (SCLIssuesTableEntry)element;
                return entry.moduleName;
            }
            @Override
            public Image getImage(Object element) {
                SCLIssuesTableEntry entry = (SCLIssuesTableEntry)element;
                return entry.error.severity == ErrorSeverity.ERROR
                        ? imageRegistry.get("error")
                        : imageRegistry.get("warning");
            }
        });
        
        column2 = new TableViewerColumn(tableViewer, SWT.NONE);
        column2.getColumn().setText("Description");
        column2.getColumn().setResizable(false);
        column2.setLabelProvider(new ColumnLabelProvider() {
            @Override
            public String getText(Object element) {
                SCLIssuesTableEntry entry = (SCLIssuesTableEntry)element;
                String description = entry.error.description; 
                int p = description.indexOf('\n');
                if(p == -1)
                    return description;
                else
                    return description.substring(0, p);
            }
            @Override
            public String getToolTipText(Object element) {
                SCLIssuesTableEntry entry = (SCLIssuesTableEntry)element;
                return entry.error.description;
            }
            @Override
            public int getToolTipTimeDisplayed(Object object) {
                return 15000;
            }
            @Override
            public int getToolTipDisplayDelayTime(Object object) {
                return 500;
            }
            @Override
            public Point getToolTipShift(Object object) {
                return new Point(0, 0);
            }
        });
        
        ControlAdapter resizeListener = new ControlAdapter() {
            @Override
            public void controlResized(ControlEvent e) {
                resizeColumns();
            }
        };
        parent.addControlListener(resizeListener);
        column1.getColumn().addControlListener(resizeListener);

        tableViewer.setInput(SCLOsgi.MODULE_REPOSITORY);
        tableViewer.addDoubleClickListener(new IDoubleClickListener() {
            @Override
            public void doubleClick(DoubleClickEvent event) {
                IStructuredSelection selection = (IStructuredSelection)event.getSelection();
                SCLIssuesTableEntry entry = (SCLIssuesTableEntry)selection.getFirstElement();
                OpenSCLDefinition.openDefinition(entry.moduleName, entry.error.location);
            }
        });
    }

    private void resizeColumns() {
        Table table = tableViewer.getTable();
        Rectangle area = parent.getClientArea();
        Point size = table.computeSize(SWT.DEFAULT, SWT.DEFAULT);
        ScrollBar vBar = table.getVerticalBar();
        int width = area.width;
        if (size.y > area.height + table.getHeaderHeight()) {
            Point vBarSize = vBar.getSize();
            width -= vBarSize.x;
        }
        Point oldSize = table.getSize();
        if (oldSize.x > area.width) {
            column2.getColumn().setWidth(Math.max(COLUMN2_MIN_WIDTH, width - column1.getColumn().getWidth()));
            table.setSize(area.width, area.height);
        } else {
            table.setSize(area.width, area.height);
            column2.getColumn().setWidth(Math.max(COLUMN2_MIN_WIDTH, width - column1.getColumn().getWidth()));
        }
    }

    @Override
    public void setFocus() {
        tableViewer.getControl().setFocus();
    }
    
    @Override
    public void dispose() {
        super.dispose();
        issuesContentProvider.dispose();
    }

}
