package org.simantics.scl.ui.issues;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.concurrent.atomic.AtomicBoolean;

import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.widgets.Control;
import org.simantics.scl.compiler.errors.CompilationError;
import org.simantics.scl.compiler.errors.DoesNotExist;
import org.simantics.scl.compiler.errors.Failable;
import org.simantics.scl.compiler.errors.Failure;
import org.simantics.scl.compiler.module.Module;
import org.simantics.scl.compiler.module.repository.ModuleRepository;
import org.simantics.scl.compiler.module.repository.UpdateListener;

import gnu.trove.map.hash.THashMap;
import gnu.trove.procedure.TObjectObjectProcedure;
import gnu.trove.procedure.TObjectProcedure;

public class SCLIssuesContentProvider implements IStructuredContentProvider {

    public static final int MAX_ISSUE_COUNT = 1000;
    
    Viewer viewer;
    ModuleRepository repository;
    boolean disposed = false;
    AtomicBoolean refreshInProgress = new AtomicBoolean(false);
    
    THashMap<String, CompilationError[]> currentFailures = new THashMap<String, CompilationError[]>();
    THashMap<String, UpdateListener> updateListeners = new THashMap<String, UpdateListener>(); 
    
    @Override
    public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
        this.viewer = viewer;
        this.repository = (ModuleRepository)newInput;
        if(newInput != null)
            listenIssues();
    }
    
    private UpdateListener getUpdateListener(String moduleName) {
        UpdateListener listener;
        synchronized(updateListeners) {
            listener = updateListeners.get(moduleName);
            if(listener == null) {
                listener = new UpdateListener() {
                    @Override
                    public void notifyAboutUpdate() {
                        if(!disposed)
                            listenModule(moduleName);
                    }
                };
                updateListeners.put(moduleName, listener);
            }
        }
        return listener;
    }
    
    private void listenModule(String moduleName) {
        if(repository == null)
            return;
        Failable<Module> result = repository.getModule(moduleName, getUpdateListener(moduleName));
        synchronized(currentFailures) {
            if(result instanceof Failure) {
                Failure failure = (Failure)result;
                currentFailures.put(moduleName, failure.errors);
            }
            else if(result == DoesNotExist.INSTANCE) {
                if(currentFailures.remove(moduleName) == null)
                    return;
            }
            else {
                CompilationError[] warnings = result.getResult().getWarnings();
                if(warnings.length == 0) {
                    if(currentFailures.remove(moduleName) == null)
                        return;
                }
                else {
                    currentFailures.put(moduleName, warnings);
                }
            }
        }
        refresh();
    }
    
    private void refresh() {
        if(!refreshInProgress.compareAndSet(false, true))
            return;
        if(viewer == null)
            return;
        Control control = viewer.getControl();
        if(control.isDisposed() || disposed)
            return;
        control.getDisplay().asyncExec(new Runnable() {
            @Override
            public void run() {
                if(control.isDisposed() || disposed)
                    return;
                refreshInProgress.set(false);
                viewer.refresh();
            }
        });
    }

    private void listenIssues() {
        new Thread() {
            public void run() {
                if(repository == null)
                    return;
                repository.getSourceRepository().forAllModules(new TObjectProcedure<String>() {
                    @Override
                    public boolean execute(String moduleName) {
                        listenModule(moduleName);
                        return true;
                    }
                });
            }
        }.start();
    }

    @Override
    public void dispose() {
        if(this.disposed)
            return;
        this.disposed = true;
        if(repository != null)
            synchronized(updateListeners) {
                updateListeners.forEachEntry(new TObjectObjectProcedure<String, UpdateListener>() {
                    @Override
                    public boolean execute(String moduleName, UpdateListener listener) {
                        listener.stopListening();
                        return true;
                    }
                });
                updateListeners.clear();
            }
    }

    @Override
    public Object[] getElements(Object inputElement) {
        ArrayList<SCLIssuesTableEntry> result = new ArrayList<SCLIssuesTableEntry>();
        synchronized(currentFailures) {
            String[] moduleNames = currentFailures.keySet().toArray(new String[currentFailures.size()]);
            Arrays.sort(moduleNames);
            for(String moduleName : moduleNames) {
                CompilationError[] errors = currentFailures.get(moduleName);
                for(CompilationError error : errors)
                    result.add(new SCLIssuesTableEntry(moduleName, error));
                if(result.size() >= MAX_ISSUE_COUNT)
                    break;
            }
        }
        Collections.sort(result);
        return result.toArray();
    }

}
