/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.scenegraph.g2d.nodes;

import java.awt.AlphaComposite;
import java.awt.Composite;
import java.awt.Graphics2D;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;

import org.simantics.scenegraph.g2d.IG2DNode;
import org.simantics.scenegraph.g2d.events.EventTypes;
import org.simantics.scenegraph.g2d.events.MouseEvent;
import org.simantics.scenegraph.g2d.events.MouseEvent.MouseDragBegin;
import org.simantics.scenegraph.utils.InitValueSupport;
import org.simantics.scenegraph.utils.NodeUtil;

public class SingleElementNode extends TransformNode implements InitValueSupport {

    private static final long serialVersionUID = -4982578347433716440L;

    private TransferableProvider transferableProvider;
    protected Composite composite;
    protected Boolean visible = Boolean.TRUE;
    protected Boolean hidden = Boolean.FALSE;

    public void setTransferableProvider(TransferableProvider transferableProvider) {
        if (transferableProvider != this.transferableProvider) {
            if (this.transferableProvider != null)
                removeEventHandler(this);
            if (transferableProvider != null)
                addEventHandler(this);
            this.transferableProvider = transferableProvider;
        }
    }

    @Override
    public boolean validate() {
        return visible && !hidden;
    }

    @SyncField("composite")
    public void setComposite(Composite composite) {
        this.composite = composite;
    }

    @SyncField("visible")
    public void setVisible(Boolean visible) {
        this.visible = visible;
    }

    public boolean isVisible() {
        return visible;
    }

    @SyncField("hidden")
    public void setHidden(Boolean hidden) {
        this.hidden = hidden;
    }

    public boolean isHidden() {
        return hidden;
    }

    @Override
    public void render(Graphics2D g) {
        if (!visible || hidden)
            return;

        Composite oldComposite = null;
        if(composite != null) {
            oldComposite = g.getComposite();
            g.setComposite(composite);
        }
        if(alphaComposite != null) {
            if (oldComposite == null)
                oldComposite = g.getComposite();
            g.setComposite(alphaComposite);
        }

        super.render(g);

        if (oldComposite != null)
            g.setComposite(oldComposite);
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append(super.toString());
        sb.append(" [");
        if (composite != null) {
            sb.append("composite=(");
            AlphaComposite ac = (AlphaComposite) composite;
            sb.append(ac.getRule());
            sb.append(", ");
            sb.append(ac.getAlpha());
            sb.append("), ");
        }
        sb.append("visible=");
        sb.append(visible);
        sb.append("]");
        return sb.toString();
    }

    protected Composite alphaComposite = null;

    @PropertySetter("alpha")
    @SyncField("alphaComposite")
    public void setAlphaComposite(Composite alphaComposite) {
        this.alphaComposite = alphaComposite;
    }

    @Override
    public void initValues() {
        alphaComposite = null;
        for (IG2DNode node : getSortedNodes()) {
            if (node instanceof InitValueSupport) {
                ((InitValueSupport) node).initValues();
            }
        }
    }

    @Override
    public void cleanup() {
        if (transferableProvider != null) {
            removeEventHandler(this);
            transferableProvider = null;
        }
        super.cleanup();
    }

    protected boolean isDragStartEvent(MouseEvent e) {
        return e.isControlDown();
    }

    protected boolean hitTest(MouseEvent event) {
        Rectangle2D bounds = super.getBoundsInLocal(true);
        if (bounds == null)
            return false;
        Point2D localPos = NodeUtil.worldToLocal(this, event.controlPosition, new Point2D.Double());
        double x = localPos.getX();
        double y = localPos.getY();
        boolean hit = bounds.contains(x, y);
        return hit;
    }

    @Override
    protected boolean mouseDragged(MouseDragBegin e) {
        if (transferableProvider == null
                || !isDragStartEvent(e)
                || !hitTest(e))
            return false;
        e.transferable = transferableProvider.create();
        return e.transferable != null;
    }

    @Override
    public int getEventMask() {
        return EventTypes.MouseDragBeginMask;
    }

}
