/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
/*
 *
 * @author Toni Kalajainen
 */
package org.simantics.scenegraph.g2d.events;

import java.awt.datatransfer.Transferable;
import java.awt.event.InputEvent;
import java.awt.geom.Point2D;

/**
 * TODO Remove hold time
 * @see Event
 * @see MouseClickEvent
 * @see MouseDragBegin
 * 
 * @author Toni Kalajainen
 */
public abstract class MouseEvent extends Event {

    private static final long serialVersionUID = -2090477028406944454L;

    public static final int   LEFT_BUTTON        = 1;
    public static final int   RIGHT_BUTTON       = 2;
    public static final int   MIDDLE_BUTTON      = 3;

    /** Time for mouse press + mouse release to be interpreted as mouse click */
    public static final long  CLICK_TIME         = 250;

    public static final int   LEFT_MASK          = 1 << (LEFT_BUTTON - 1);
    public static final int   RIGHT_MASK         = 1 << (RIGHT_BUTTON - 1);
    public static final int   MIDDLE_MASK        = 1 << (MIDDLE_BUTTON - 1);
    public static final int   ALL_BUTTONS_MASK   = LEFT_MASK | RIGHT_MASK | MIDDLE_MASK;

    public static final int   CTRL_MASK          = InputEvent.CTRL_DOWN_MASK;
    public static final int   ALT_MASK           = InputEvent.ALT_DOWN_MASK;
    public static final int   ALT_GRAPH_MASK     = InputEvent.ALT_GRAPH_DOWN_MASK;
    public static final int   SHIFT_MASK         = InputEvent.SHIFT_DOWN_MASK;
    public static final int   ALL_MODIFIERS_MASK = CTRL_MASK | ALT_MASK | ALT_GRAPH_MASK | SHIFT_MASK;

    /**
     * The identifier of the mouse of this event.
     * 0 = the main mouse.
     */
    public final int mouseId;

    /** Position in the coordinate system of the UI component */
    public final Point2D controlPosition;

    /** Position in the coordinate system of the screen */
    public final Point2D screenPosition;

    /**
     * The status of mouse buttons
     * @see #LEFT_MASK
     * @see #MIDDLE_MASK
     * @see #RIGHT_MASK
     */
    public final int buttons;

    /**
     * The state of the keyboard modifier keys at the time the event was
     * generated.
     * 
     * @see #ALT_MASK
     * @see #ALT_GRAPH_MASK
     * @see #CTRL_MASK
     * @see #SHIFT_MASK
     */
    public int stateMask;

    private MouseEvent(Object context, long time, int mouseId, int buttons, int stateMask,
            Point2D controlPosition, Point2D screenPosition) {
        super(context, time);
        this.mouseId = mouseId;
        this.buttons = buttons;
        this.controlPosition = controlPosition;
        this.screenPosition = screenPosition;
        this.stateMask = stateMask;
    }

    public boolean hasAnyButton(int mask) {
        return (buttons & mask) != 0;
    }

    public boolean hasAllButtons(int mask) {
        return (buttons & mask) == mask;
    }

    public boolean hasAnyModifier(int mask) {
        return (stateMask & mask) != 0;
    }

    public boolean hasAllModifiers(int mask) {
        return (stateMask & mask) == mask;
    }

    /**
     * Returns whether or not the Shift modifier is down on this event.
     */
    public boolean isShiftDown() {
        return (stateMask & MouseEvent.SHIFT_MASK) != 0;
    }

    /**
     * Returns whether or not the Control modifier is down on this event.
     */
    public boolean isControlDown() {
        return (stateMask & MouseEvent.CTRL_MASK) != 0;
    }

    /**
     * Returns whether or not the Alt modifier is down on this event.
     */
    public boolean isAltDown() {
        return (stateMask & MouseEvent.ALT_MASK) != 0;
    }

    /**
     * Returns whether or not the Alt Graph modifier is down on this event.
     */
    public boolean isAltGraphDown() {
        return (stateMask & MouseEvent.ALT_GRAPH_MASK) != 0;
    }

    public abstract static class MouseButtonEvent extends MouseEvent {

        private static final long serialVersionUID = 3540032494506535841L;

        /**
         * button of this event
         */
        public final int button;

        public MouseButtonEvent(Object context, long time, int mouseId, int buttons, int stateMask,
                int button, Point2D controlPosition, Point2D screenPosition) {
            super(context, time, mouseId, buttons, stateMask, controlPosition,screenPosition);
            this.button  = button;
        }
    }

    public static class MouseWheelMovedEvent extends MouseEvent {

        private static final long serialVersionUID = -7896477913481842708L;

        /**
         * Constant representing scrolling by "units" (like scrolling with the
         * arrow keys)
         * 
         * @see #getScrollType
         */
        public static final int WHEEL_UNIT_SCROLL = 0;

        /**
         * Constant representing scrolling by a "block" (like scrolling
         * with page-up, page-down keys)
         *
         * @see #getScrollType
         */
        public static final int WHEEL_BLOCK_SCROLL = 1;

        /**
         * Indicates what sort of scrolling should take place in response to this
         * event, based on platform settings.  Legal values are:
         * <ul>
         * <li> WHEEL_UNIT_SCROLL
         * <li> WHEEL_BLOCK_SCROLL
         * </ul>
         * 
         * @see #getScrollType
         */
        public final int scrollType;

        /**
         * Only valid for scrollType WHEEL_UNIT_SCROLL.
         * Indicates number of units that should be scrolled per
         * click of mouse wheel rotation, based on platform settings.
         *
         * @see #getScrollAmount
         * @see #getScrollType
         */
        public final int scrollAmount;

        /**
         * Indicates how far the mouse wheel was rotated.
         *
         * @see #getWheelRotation
         */
        public final int wheelRotation;

        public MouseWheelMovedEvent(Object context, long time, int mouseId, int buttons, int stateMask,
                Point2D controlPosition, Point2D screenPosition, int scrollType, int scrollAmount,
                int wheelRotation) {
            super(context, time, mouseId, buttons, stateMask, controlPosition,screenPosition);
            this.scrollType = scrollType;
            this.scrollAmount = scrollAmount;
            this.wheelRotation = wheelRotation;
        }

    }

    public static class MouseButtonPressedEvent extends MouseButtonEvent {

        private static final long serialVersionUID = -4687294674299429690L;

        public MouseButtonPressedEvent(Object context, long time, int mouseId, int buttons, int stateMask,
                int button, Point2D controlPosition, Point2D screenPosition) {
            super(context, time, mouseId, buttons, stateMask, button, controlPosition,screenPosition);
        }

    }

    public static class MouseButtonReleasedEvent extends MouseButtonEvent {

        private static final long serialVersionUID = -2303672225339491858L;

        /** Time in milliseconds how long the button was held down */
        public final long holdTime;

        public MouseButtonReleasedEvent(Object context, long time,
                int mouseId, int buttons, int stateMask, int button, long holdTime,
                Point2D controlPosition, Point2D screenPosition) {
            super(context, time, mouseId, buttons, stateMask, button, controlPosition, screenPosition);
            this.holdTime = holdTime;
        }
    }

    public static class MouseMovedEvent extends MouseEvent {
        private static final long serialVersionUID = 1463958776335678155L;

        public MouseMovedEvent(Object context, long time, int mouseId, int buttons, int stateMask,
                Point2D controlPosition, Point2D screenPosition) {
            super(context, time, mouseId, buttons, stateMask, controlPosition, screenPosition);
        }
    }

    public static class MouseEnterEvent extends MouseEvent {
        private static final long serialVersionUID = 6074648747865556588L;

        public MouseEnterEvent(Object context, long time, int mouseId, int buttons, int stateMask,
                Point2D controlPosition, Point2D screenPosition) {
            super(context, time, mouseId, buttons, stateMask, controlPosition, screenPosition);
        }
    }

    public static class MouseExitEvent extends MouseEvent {
        private static final long serialVersionUID = 8596801599996844789L;

        public MouseExitEvent(Object context, long time, int mouseId, int buttons, int stateMask,
                Point2D controlPosition, Point2D screenPosition) {
            super(context, time, mouseId, buttons, stateMask, controlPosition, screenPosition);
        }
    }

    public static class MouseDoubleClickedEvent extends MouseButtonEvent {
        private static final long serialVersionUID = -8046967155607105912L;

        public MouseDoubleClickedEvent(Object context, long time, int mouseId, int buttons, int stateMask,
                int button, Point2D controlPosition, Point2D screenPosition) {
            super(context, time, mouseId, buttons, stateMask, button, controlPosition, screenPosition);
        }
    }

    public static class MouseClickEvent extends MouseButtonEvent {
        private static final long serialVersionUID = -1737712792090986607L;
        /**
         * Indicates the number of quick consecutive clicks of
         * a mouse button.
         */
        public final int clickCount;
    
        public MouseClickEvent(Object context, long time, int mouseId, int buttons, int stateMask,
                int button, int clickCount, Point2D controlPosition, Point2D screenPosition) {
            super(context, time, mouseId, buttons, stateMask, button, controlPosition, screenPosition);
            this.clickCount = clickCount;
        }
    }

    public static class MouseDragBegin extends MouseButtonEvent {
        private static final long serialVersionUID = 3882552912240236450L;
        public final Point2D startCanvasPos;
        public final Point2D startControlPos;
        public Transferable transferable = null;
        public MouseDragBegin(Object context, long time, int mouseId, int buttons, int stateMask,
                int button, Point2D startCanvasPos, Point2D startControlPos, Point2D controlPosition, Point2D screenPosition) {
            super(context, time, mouseId, buttons, stateMask, button, controlPosition, screenPosition);
            this.startCanvasPos = startCanvasPos;
            this.startControlPos = startControlPos;
        }
    }

    /**
     * Spans button bit mask into individual button ids (starting from 1)
     * @param buttons
     * @return
     */
    public static int[] getButtonIds(int buttons)
    {
        // calculate 1 bits
        int size = 0;
        for (int i=0; i<32; i++)
            if ((buttons & (1<<i))!=0)
                size++;
        int result[] = new int[size];
        int j=0;
        for (int i=0; i<32; i++)
            if ((buttons & (1<<i))!=0)
                result[j++] = i+1;
        return result;
    }

    @Override
    public String toString() {
        return super.toString() + "[mouse#=" + mouseId + ", control pos=" + controlPosition + ", screen pos="
                + screenPosition + ", buttons=" + Integer.toBinaryString(buttons) + ", stateMask=0x"
                + Integer.toHexString(stateMask) + "]";
    }

}
