/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.issues.ui.handler;

import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;

import org.eclipse.core.commands.AbstractHandler;
import org.eclipse.core.commands.ExecutionEvent;
import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorReference;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.simantics.Simantics;
import org.simantics.browsing.ui.common.ErrorLogger;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.request.ActiveModels;
import org.simantics.db.layer0.request.PossibleModel;
import org.simantics.db.layer0.util.ModelTransferableGraphSourceRequest;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.issues.common.BatchIssueSource;
import org.simantics.issues.common.BatchIssueValidationContext;
import org.simantics.issues.common.SelectedModelBatchIssueSources;
import org.simantics.modeling.ModelingResources;
import org.simantics.ui.workbench.IResourceEditorInput;
import org.simantics.utils.ui.ExceptionUtils;

/**
 * @author Tuukka Lehtonen
 */
public class UpdateIssuesForOpenDiagrams extends AbstractHandler {

    @Override
    public Object execute(ExecutionEvent event) throws ExecutionException {
        Runnable postValidation = null;
        try {
            update(postValidation);
        } catch (DatabaseException e) {
            ErrorLogger.defaultLogError(e);
        } catch (PartInitException e) {
            ErrorLogger.defaultLogError(e);
        }
        return null;
    }

    public static void update(Runnable postValidation) throws DatabaseException, PartInitException {
        final List<IResourceEditorInput> inputs = new ArrayList<>();
        IWorkbenchWindow[] windows = PlatformUI.getWorkbench().getWorkbenchWindows();
        for (IWorkbenchWindow wb : windows) {
            IWorkbenchPage pages[] = wb.getPages();
            for (IWorkbenchPage page : pages) {
                for (IEditorReference ref : page.getEditorReferences()) {
                    IEditorInput input = ref.getEditorInput();
                    if (input instanceof IResourceEditorInput) {
                        IResourceEditorInput in = (IResourceEditorInput) input;
                        inputs.add(in);
                    }
                }
            }
        }

        final Set<Resource> compositesToValidate = new TreeSet<>();
        final Collection<BatchIssueSource> validations = new ArrayList<>();
        final Collection<Resource> activeModels = new ArrayList<>();
    	final BatchIssueValidationContext context = new BatchIssueValidationContext();

        try {
            PlatformUI.getWorkbench().getProgressService().busyCursorWhile(new IRunnableWithProgress() {
                @Override
                public void run(final IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
                    try {
                        Resource project = Simantics.getProjectResource();
                        Session session = Simantics.getSession();

                        activeModels.addAll(session.syncRequest(new ActiveModels(project)));
                        if(activeModels.size() != 1) return;

                        RunActiveValidations.toBatchIssueSources(session,
                                session.syncRequest(new SelectedModelBatchIssueSources(activeModels.iterator().next())),
                                validations);

                        session.syncRequest(new ReadRequest() {
                            @Override
                            public void run(ReadGraph graph) throws DatabaseException {
                                DiagramResource DIA = DiagramResource.getInstance(graph);
                                ModelingResources MOD = ModelingResources.getInstance(graph);
                                for (IResourceEditorInput input : inputs) {
                                    Resource in = input.getResource();
                                    if (graph.isInstanceOf(in, DIA.Diagram)) {
                                        Resource composite = graph.getPossibleObject(in, MOD.DiagramToComposite);
                                        if (composite != null) {
                                            Resource model = graph.sync(new PossibleModel(composite));
                                            if (activeModels.contains(model)) {
                                                compositesToValidate.add(composite);
                                                //System.out.println("TODO: re-validate composite " + NameUtils.getSafeLabel(graph, composite));
                                            }
                                        }
                                    }
                                }
                                
                                context.contexts = compositesToValidate;
                                if(!compositesToValidate.isEmpty()) {
                                	ModelTransferableGraphSourceRequest.getDomainOnly(graph, monitor, compositesToValidate);
                                }
                                
                            }
                        });
                    } catch (DatabaseException e) {
                        throw new InvocationTargetException(e);
                    } finally {
                        monitor.done();
                    }
                }
            });
        } catch (InvocationTargetException e) {
            ExceptionUtils.logAndShowError(e.getTargetException());
            return;
        } catch (InterruptedException e) {
            // Operation cancelled, ignore.
            return;
        }

        if (!context.contexts.isEmpty() && activeModels.size() == 1) {
            RunActiveValidations.run(postValidation, validations, context);
        }
        
    }

}
