/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.issues.ui.handler;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.eclipse.jface.action.ActionContributionItem;
import org.eclipse.jface.action.ContributionItem;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IContributionItem;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.SelectionHints;
import org.simantics.issues.Severity;
import org.simantics.issues.common.IssueConstants;
import org.simantics.issues.common.IssueUtils;
import org.simantics.issues.ontology.IssueResource;
import org.simantics.issues.ui.internal.Activator;
import org.simantics.ui.contribution.DynamicMenuContribution;
import org.simantics.utils.ui.ErrorLogger;
import org.simantics.utils.ui.ISelectionUtils;

/**
 * @author Tuukka Lehtonen
 */
public class MenuActions extends DynamicMenuContribution {

    @Override
    protected Object[] getSelectedObjects() {
        ISelection sel = getSelection();
        List<Resource> resources = ISelectionUtils.getPossibleKeys(sel, SelectionHints.KEY_MAIN, Resource.class);
        return resources.toArray();
    }

    List<Resource> toResources(Object[] array) {
        Resource[] a = new Resource[array.length];
        for (int i = 0; i < array.length; ++i)
            a[i] = (Resource) array[i];
        return Arrays.asList(a);
    }

    @Override
    protected IContributionItem[] getContributionItems(ReadGraph graph, Object[] selection) throws DatabaseException {
        if (selection.length == 0)
            return new IContributionItem[0];

        List<Resource> input = toResources(selection);

        @SuppressWarnings("unused")
        boolean issues = false;
        @SuppressWarnings("unused")
        boolean userIssues = false;

        IssueResource ISSUE = IssueResource.getInstance(graph);
        int issueCount = 0;
        int hiddenCount = 0; 
        int resolvedCount = 0;
        int userIssueCount = 0;
        for (Resource r : input) {
            boolean isIssue = graph.isInstanceOf(r, ISSUE.Issue);
            issues |= isIssue;
            if (isIssue) {
                ++issueCount;
                if (graph.hasStatement(r, ISSUE.Hidden))
                    ++hiddenCount;
                if (graph.hasStatement(r, ISSUE.Resolved))
                    ++resolvedCount;
            }

            boolean userIssue = graph.hasStatement(r, ISSUE.UserIssue);
            userIssues |= userIssue;
            if (userIssue) {
                ++userIssueCount;
            }
        }
        boolean allHidden = hiddenCount == selection.length;
        boolean allResolved = resolvedCount == selection.length;
        boolean allUserIssues = userIssueCount == selection.length;
        boolean allIssueSelection = issueCount == selection.length;

        List<IContributionItem> actions = new ArrayList<IContributionItem>(2); 
        if (!allHidden && allIssueSelection)
            actions.add(new ActionContributionItem(hideAction(input)));
        if (hiddenCount > 0 || allHidden && allIssueSelection)
            actions.add(new ActionContributionItem(unhideAction(input)));
        if (!allResolved && allIssueSelection)
            actions.add(new ActionContributionItem(resolveAction(input)));
        if (resolvedCount > 0 || allResolved && allIssueSelection)
            actions.add(new ActionContributionItem(unresolveAction(input)));
        if (allUserIssues)
            actions.add(setSeverityAction(input));

        return actions.toArray(new IContributionItem[actions.size()]);
    }

    private IAction unhideAction(List<Resource> input) {
        return tagAction("Unhide", Activator.UNHIDE_ICON, IssueResource.URIs.Hidden, false, input);
    }

    private IAction hideAction(List<Resource> input) {
        return tagAction("Hide", Activator.HIDE_ICON, IssueResource.URIs.Hidden, true, input);
    }

    private IAction resolveAction(List<Resource> input) {
        return tagAction("Mark Resolved", Activator.RESOLVE_ICON, IssueResource.URIs.Resolved, true, input);
    }

    private IAction unresolveAction(List<Resource> input) {
        return tagAction("Mark Unresolved", Activator.UNRESOLVE_ICON, IssueResource.URIs.Resolved, false, input);
    }

    private IAction tagAction(String label, ImageDescriptor image, String tagURI, boolean tag, List<Resource> input) {
        return new TagAction(label, image, IssueConstants.ISSUE_VG, tagURI, tag, input);
        //return new TagAction(label, image, null, tagURI, tag, input);
    }

    private IContributionItem setSeverityAction(final List<Resource> userIssues) {
        return new ContributionItem() {
            @Override
            public void fill(Menu menu, int index) {
                MenuItem setSeverityItem = new MenuItem(menu, SWT.CASCADE);
                setSeverityItem.setText("Set Severity");
                Menu setSeverity = new Menu(menu);
                setSeverityItem.setMenu(setSeverity);
                for (final Severity sev : Severity.values()) {
                    if (sev.equals(Severity.UNKNOWN))
                        continue;

                    MenuItem item = new MenuItem(setSeverity, SWT.PUSH);
                    item.setText(sev.toString().toLowerCase());
                    item.setImage(Activator.getDefault().getImageRegistry().get(sev.toString() + "-full"));
                    item.addSelectionListener(new SelectionAdapter() {
                        @Override
                        public void widgetSelected(SelectionEvent e) {
                            setSeverity(userIssues, sev);
                        }
                    });
                }
            }
        };
    }

    private void setSeverity(final List<Resource> userIssues, final Severity severity) {
        Simantics.getSession().asyncRequest(new WriteRequest() {
            @Override
            public void perform(WriteGraph graph) throws DatabaseException {
                IssueResource ISSUE = IssueResource.getInstance(graph);
                Resource sev = IssueUtils.toSeverityResource(ISSUE, severity);
                if (sev != null) {
                    for (Resource issue : userIssues) {
                        if (!graph.hasStatement(issue, ISSUE.Issue_HasSeverity, sev)) {
                            graph.deny(issue, ISSUE.Issue_HasSeverity);
                            graph.claim(issue, ISSUE.Issue_HasSeverity, null, sev);
                        }
                    }
                } else {
                    for (Resource issue : userIssues)
                        graph.deny(issue, ISSUE.Issue_HasSeverity);
                }
            }
        }, e -> {
            if (e != null)
                ErrorLogger.defaultLogError(e);
        });
    }

}