/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.issues.ui.handler;

import java.io.File;
import java.io.PrintStream;
import java.lang.reflect.InvocationTargetException;
import java.util.Collection;

import org.eclipse.core.commands.AbstractHandler;
import org.eclipse.core.commands.ExecutionEvent;
import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.SubMonitor;
import org.eclipse.core.runtime.preferences.InstanceScope;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.handlers.HandlerUtil;
import org.osgi.service.prefs.BackingStoreException;
import org.osgi.service.prefs.Preferences;
import org.simantics.Simantics;
import org.simantics.browsing.ui.common.ErrorLogger;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.ObjectsWithType;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.request.Read;
import org.simantics.issues.common.AllVisibleIssues;
import org.simantics.issues.ui.internal.Activator;
import org.simantics.operation.Layer0X;
import org.simantics.simulation.ontology.SimulationResource;
import org.simantics.utils.DataContainer;
import org.simantics.utils.FileUtils;
import org.simantics.utils.strings.StringUtils;
import org.simantics.utils.ui.ExceptionUtils;

/**
 * @author Tuukka Lehtonen
 */
public class ExportIssuesAsCsv extends AbstractHandler {

    private static final String PROP_LAST_VALIDATION_REPORT_PATH= "validation.report.path";

    @Override
    public Object execute(ExecutionEvent event) throws ExecutionException {
        IWorkbenchWindow window = HandlerUtil.getActiveWorkbenchWindow(event);

        try {
            String fileName = generateFileName();
            validate(window, fileName);
        } catch (DatabaseException e) {
            ErrorLogger.defaultLogError(e);
        }

        return null;
    }

    private String generateFileName() throws DatabaseException {
        String generatedName = Simantics.getSession().syncRequest(new Read<String>() {
            @Override
            public String perform(ReadGraph graph) throws DatabaseException {
                Layer0X L0X = Layer0X.getInstance(graph);
                SimulationResource SIMU = SimulationResource.getInstance(graph);
                for (Resource model : graph.syncRequest(new ObjectsWithType(Simantics.getProjectResource(), L0X.Activates, SIMU.Model))) {
                    return NameUtils.getSafeName(graph, model) + ".txt";
                }
                return "issues.txt";
            }
        });

        if (!FileUtils.isValidFileName(generatedName))
            generatedName = (String) Bindings.STR_VARIANT.createUnchecked(Bindings.STRING, generatedName);

        return generatedName;
    }

    public void validate(IWorkbenchWindow window, String fileName) {
        Preferences prefs = InstanceScope.INSTANCE.getNode(Activator.PLUGIN_ID);
        String lastReportPath = prefs.get(PROP_LAST_VALIDATION_REPORT_PATH, null);

        // Query for output path
        Shell parentShell = null;
        if (window != null)
            parentShell = window.getShell();

        final DataContainer<PrintStream> externalOutput = new DataContainer<PrintStream>();

        FileDialog fd = new FileDialog(parentShell, SWT.SAVE);
        fd.setText("Select Validation Output");
        fd.setFilterExtensions(new String[] { "*.txt", "*.*" });
        fd.setFilterNames(new String[] { "Comma-Separated Values (*.txt)", "All Files (*.*)" });
        if (lastReportPath != null)
            fd.setFilterPath(lastReportPath);
        fd.setFileName(fileName);
        final String path = fd.open();
        if (path != null) {
            prefs.put(PROP_LAST_VALIDATION_REPORT_PATH, path);
            try {
                prefs.flush();
            } catch (BackingStoreException e) {
                ExceptionUtils.logError(e);
            }
        } else {
//            boolean result = MessageDialog.openQuestion(parentShell, "Print to stdout?", "Would you like to perform model validation and print the results into stdout?");
//            if (!result)
//                return;
//            externalOutput.set(System.out);
            return;
        }

        try {
            window.getWorkbench().getProgressService().busyCursorWhile(new IRunnableWithProgress() {
                @Override
                public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
                    try {
                        if (externalOutput.get() == null)
                            externalOutput.set(new PrintStream(new File(path)));

                        final SubMonitor progress = SubMonitor.convert(monitor, "Export issues", IProgressMonitor.UNKNOWN);
                        Simantics.getSession().syncRequest(new ReadRequest() {
                            @Override
                            public void run(ReadGraph graph) throws DatabaseException {
                                PrintStream out = externalOutput.get();

                                Collection<Variable> activeIssues = graph.syncRequest(new AllVisibleIssues(Simantics.getProjectResource()));
                                out.println("# Exported issues (" + activeIssues.size() + ")");
                                for (Variable issue : activeIssues) {
                                    String description = StringUtils.safeString( (String) issue.getPossiblePropertyValue(graph, "HasDescription") );
                                    String severity = StringUtils.safeString( (String) issue.getPossiblePropertyValue(graph, "severity") );
                                    String resource = StringUtils.safeString( (String) issue.getPossiblePropertyValue(graph, "resource") );
                                    String path = StringUtils.safeString( (String) issue.getPossiblePropertyValue(graph, "path") );
                                    out.println(description + ";" + severity + ";" + resource + ";" + path);
                                    progress.worked(1);
                                }
                            }
                        });
                    } catch (Exception e) {
                        throw new InvocationTargetException(e);
                    } finally {
                        monitor.done();
                        if (externalOutput.get() != System.out)
                            FileUtils.uncheckedClose(externalOutput.get());
                    }
                }
            });
        } catch (InvocationTargetException e) {
            ExceptionUtils.logAndShowError(e.getTargetException());
        } catch (InterruptedException e) {
            // Operation cancelled, ignore.
        }
    }

}