/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.issues.ui.handler;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.eclipse.core.commands.AbstractHandler;
import org.eclipse.core.commands.ExecutionEvent;
import org.eclipse.core.commands.ExecutionException;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.viewers.CheckStateChangedEvent;
import org.eclipse.jface.viewers.CheckboxTableViewer;
import org.eclipse.jface.viewers.ICheckStateListener;
import org.eclipse.jface.viewers.ICheckStateProvider;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.ui.PlatformUI;
import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.NamedResource;
import org.simantics.db.common.request.PossibleObjectWithType;
import org.simantics.db.common.request.UniqueRead;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.request.PossibleActiveModel;
import org.simantics.db.layer0.util.RemoverUtil;
import org.simantics.issues.common.IssueUtils;
import org.simantics.issues.ontology.IssueResource;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ModelingUtils;
import org.simantics.utils.ui.ErrorLogger;
import org.simantics.utils.ui.dialogs.ListDialog;

/**
 * @author Antti Villberg
 */
public class ConfigureIssueSources extends AbstractHandler {

	private static class IssueSourceEntry extends NamedResource {
		private boolean checked;
		public IssueSourceEntry(String name, Resource resource, boolean checked) {
			super(name, resource);
			this.checked = checked;
		}
		public boolean isChecked() {
			return checked;
		}
		public void setChecked(boolean value) {
			checked = value;
		}
	}
	
    @Override
    public Object execute(ExecutionEvent event) throws ExecutionException {

      try {

    	final List<IssueSourceEntry> sources = Simantics.getSession().syncRequest(new UniqueRead<List<IssueSourceEntry>>() {

			@Override
			public List<IssueSourceEntry> perform(ReadGraph graph) throws DatabaseException {
				
				Resource activeModel = graph.syncRequest(new PossibleActiveModel(Simantics.getProjectResource()));
				if(activeModel == null) return Collections.emptyList();
				
				List<IssueSourceEntry> result = new ArrayList<IssueSourceEntry>();
				Layer0 L0 = Layer0.getInstance(graph);
				IssueResource ISSUE = IssueResource.getInstance(graph);
				for(Resource type : ModelingUtils.searchByType(graph, activeModel, ISSUE.IssueSourceType)) {
					String name = NameUtils.getSafeLabel(graph, type);
					boolean exists = graph.syncRequest(new PossibleObjectWithType(activeModel, L0.ConsistsOf, type)) != null;
					boolean deprecated = graph.hasStatement(type, L0.Deprecated);
					if(!exists && deprecated) continue;
					result.add(new IssueSourceEntry(name, type, exists));
				}
				return result;
			}
    		
    	});
    	
    	 Shell shell = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell();
         ListDialog<IssueSourceEntry> dialog = new ListDialog<IssueSourceEntry>(
                 shell, sources,
                 "Select available issue sources",
                 "Selected sources will be used and existing deselected sources will be removed.") {
        	 
        	    protected CheckboxTableViewer createViewer(Composite composite) {
        	    	CheckboxTableViewer viewer = CheckboxTableViewer.newCheckList(
        	    	        composite, SWT.V_SCROLL | SWT.H_SCROLL | SWT.BORDER);
        	    	final Table table = (Table)viewer.getControl();
        	    	viewer.setCheckStateProvider(new ICheckStateProvider() {
						
						@Override
						public boolean isGrayed(Object arg0) {
							return false;
						}
						
						@Override
						public boolean isChecked(Object arg0) {
							IssueSourceEntry entry = (IssueSourceEntry)arg0;
							return entry.isChecked();
						}
						
					});
        	    	viewer.addCheckStateListener(new ICheckStateListener() {
						
						@Override
						public void checkStateChanged(CheckStateChangedEvent arg0) {
							IssueSourceEntry entry = (IssueSourceEntry)arg0.getElement();
							entry.setChecked(arg0.getChecked());
						}
					});
        	    	table.addSelectionListener(new SelectionListener () {
        	    		@Override
        	    		public void widgetSelected(SelectionEvent e) {
        	    			table.deselectAll();
        	    		}
        	    		@Override
        	    		public void widgetDefaultSelected(SelectionEvent e) {}
        	    	});
        	    	return viewer;
        	    }

         };
         int result = dialog.open();
         if (result != Dialog.OK)
             return null;
    	
     	Simantics.getSession().syncRequest(new WriteRequest() {

			@Override
			public void perform(WriteGraph graph) throws DatabaseException {
				
				Resource activeModel = graph.syncRequest(new PossibleActiveModel(Simantics.getProjectResource()));
				if(activeModel == null) return;
				
				Layer0 L0 = Layer0.getInstance(graph);

				for(IssueSourceEntry entry : sources) {

					Resource existing = graph.syncRequest(new PossibleObjectWithType(activeModel, L0.ConsistsOf, entry.getResource())); 
					
					if(existing == null && entry.isChecked()) {
						String name = NameUtils.getSafeLabel(graph, entry.getResource());
						IssueUtils.addIssueSource(graph, activeModel, entry.getResource(), name);
					}
					
					if(existing != null && !entry.isChecked()) {
						RemoverUtil.remove(graph, existing);
					}
					
				}
				
			}
     		
     	});
         
//        try {
//            PlatformUI.getWorkbench().getProgressService().busyCursorWhile(new IRunnableWithProgress() {
//                @Override
//                public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
//                    try {
//                        purgeResolvedIssues(monitor);
//                    } catch (DatabaseException e) {
//                        throw new InvocationTargetException(e);
//                    } finally {
//                        monitor.done();
//                    }
//                }
//            });
//        } catch (InvocationTargetException e) {
//            ErrorLogger.defaultLogError(e);
        } catch (DatabaseException e) {
            ErrorLogger.defaultLogError(e);
        }
        return null;
    }

//    private void purgeResolvedIssues(IProgressMonitor monitor) throws DatabaseException {
//        Session session = Simantics.getSession();
//        final Resource project = Simantics.getProjectResource();
//        if (project == null)
//            return;
//
//        final SubMonitor mon = SubMonitor.convert(monitor, "Purging resolved issues...", 100);
//
//        session.syncRequest(new DelayedWriteRequest() {
//            @Override
//            public void perform(WriteGraph graph) throws DatabaseException {
//                graph.markUndoPoint();
//                IssueResource ISSUE = IssueResource.getInstance(graph);
//                Set<Resource> toBeRemoved = new HashSet<Resource>();
//                Map<Resource, Boolean> sourceIsContinuous = new THashMap<Resource, Boolean>(); 
//                for (Resource activeIssue : graph.syncRequest(new AllActiveIssues(project))) {
//                    if (graph.hasStatement(activeIssue, ISSUE.Resolved)) {
//                        Resource managedBy = graph.getPossibleObject(activeIssue, ISSUE.IssueSource_Manages_Inverse);
//                        if (managedBy != null) {
//                            Boolean isContinuous = sourceIsContinuous.get(managedBy);
//                            if (isContinuous == null) {
//                                isContinuous = graph.isInstanceOf(managedBy, ISSUE.ContinuousIssueSource);
//                                sourceIsContinuous.put(managedBy, isContinuous);
//                            }
//                            if (isContinuous)
//                                continue;
//                        }
//                        toBeRemoved.add(activeIssue);
//                    }
//                }
//
//                mon.setTaskName("Purging " + toBeRemoved.size() + " resolved batch issues...");
//                mon.setWorkRemaining(toBeRemoved.size());
//                StringBuilder sb = new StringBuilder();
//                sb.append("Purged " + toBeRemoved.size() + " resolved batch issue(s)");
//                for (Resource remove : toBeRemoved) {
//                    //sb.append(NameUtils.getSafeLabel(graph, remove) + " ");
//                    RemoverUtil.remove(graph, remove);
//                    mon.worked(1);
//                }
//                Layer0Utils.addCommentMetadata(graph, sb.toString());
//            }
//        });
//    }

}
