/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.impl.query;

import gnu.trove.procedure.TIntProcedure;

import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;
import java.util.Set;

import org.simantics.db.Resource;
import org.simantics.db.ResourceSet;
import org.simantics.db.impl.ResourceImpl;
import org.simantics.db.impl.support.ResourceSupport;


final public class IntSet implements ResourceSet {

    final private ResourceSupport support;

    public int[] data;

    /** the index after the last entry in the list */
    public int sizeOrData;

    /** the default capacity for new lists */
    protected static final int DEFAULT_CAPACITY = 3;

    public static final int NO_DATA = -1;

    public IntSet() {
        support = null;
        data = null;
        sizeOrData = NO_DATA;
    }

    public IntSet(QuerySupport support) {
        this.support = support.getSupport();
        data = null;
        sizeOrData = NO_DATA;
    }

    public IntSet(QuerySupport support, int value) {
        this.support = support.getSupport();
        data = null;
        sizeOrData = value;
    }

    @Override
    public int hashCode() {
    	return 31 * sizeOrData + 41 * Arrays.hashCode(data);
    }
    
    @Override
    public boolean equals(Object object) {
        if (this == object)
            return true;
        else if (object == null)
            return false;
        else if (IntSet.class != object.getClass())
            return false;
        IntSet r = (IntSet)object;
//        System.out.println("equals " + this + " vs. " + r);
        return sizeOrData == r.sizeOrData && Arrays.equals(data, r.data);
    }


    /**
     * Returns the number of values in the list.
     *
     * @return the number of values in the list.
     */
    public int size() {
        return data != null ? sizeOrData : (sizeOrData != NO_DATA ? 1 : 0);
    }

    /**
     * Tests whether this list contains any values.
     *
     * @return true if the list is empty.
     */
    public boolean isEmpty() {
        return sizeOrData == NO_DATA;
    }

    // modifying

    /**
     * Adds <tt>val</tt> to the end of the list, growing as needed.
     *
     * @param val an <code>int</code> value
     */
    public boolean add(int val) {
        if(data == null) {
            if(sizeOrData == val) return false;
            if(sizeOrData == NO_DATA) {
                sizeOrData = val;
                return true;
            } else {
                data = new int[DEFAULT_CAPACITY];
                data[0] = sizeOrData;
                data[1] = val;
                sizeOrData = 2;
                return true;
            }
        } else {
            for(int i=0;i<sizeOrData;i++) if(data[i] == val) return false;
            if(data.length == sizeOrData) {
                int newCap = data.length << 1;
                int[] tmp = new int[newCap];
                System.arraycopy(data, 0, tmp, 0, data.length);
                data = tmp;
                data[sizeOrData++] = val;
            } else {
                data[sizeOrData++] = val;
            }
            return true;
        }
    }

    public void forEach(TIntProcedure proc) {
        if(data == null) {
            if(sizeOrData != NO_DATA) proc.execute(sizeOrData);
        } else {
            for(int i = 0;i < sizeOrData ; i++) proc.execute(data[i]);
        }
    }
    
    public boolean contains(int val) {
        if(data == null) {
            return sizeOrData == val;
        } else {
            for(int i = 0;i < sizeOrData ; i++) if(data[i] == val) return true;
        }
        return false;
    }

    public void trim() {
        if(data != null && sizeOrData < data.length) {
            int newCap = sizeOrData;
            int[] tmp = new int[newCap];
            System.arraycopy(data, 0, tmp, 0, newCap);
            data = tmp;
            sizeOrData = newCap;
        }
    }

    @Override
    public boolean add(Resource e) {
    	return add(((ResourceImpl)e).id);
    }

    @Override
    public boolean addAll(Collection<? extends Resource> c) {
    	boolean ret = false;
        for (Resource r : c) {
        	if (add(r))
        		ret = true;
        }
        return ret;
    }

    @Override
    public void clear() {
        sizeOrData = NO_DATA;
        data = null;
    }

    @Override
    public boolean contains(Object o) {
    	assert(o != null);
        return contains(((ResourceImpl)o).id);
    }

    @Override
    public boolean containsAll(Collection<?> c) {
    	for (Object o : c) {
    		if (!contains(o))
    			return false;
    	}
    	return true;
    }

    @Override
    public Iterator<Resource> iterator() {

        class ArraySetIterator implements Iterator<Resource> {

            int next = 0;

            @Override
            public boolean hasNext() {
                return next < size();
            }

            @Override
            public Resource next() {
                if(size() == 1) {
                    next++;
                    return new ResourceImpl(support, sizeOrData);
                } else {
                    return new ResourceImpl(support, data[next++]);
                }
            }

            @Override
            public void remove() {
                throw new UnsupportedOperationException();
            }

        }

        return new ArraySetIterator();

    }

    @Override
    public boolean remove(Object o) {
        throw new UnsupportedOperationException();
    }

    @Override
    public boolean removeAll(Collection<?> c) {
        throw new UnsupportedOperationException();
    }

    @Override
    public boolean retainAll(Collection<?> c) {
        throw new UnsupportedOperationException();
    }

    @Override
    public Object[] toArray() {
        throw new UnsupportedOperationException();
    }

    @Override
    public <T> T[] toArray(T[] a) {
        throw new UnsupportedOperationException();
    }

    @Override
    public String toString() {
        return "IntSet " + sizeOrData + " " + Arrays.toString(data);
    }
    
    @Override
    public boolean disjoint(Set<Resource> other) {
        if(other instanceof ResourceSet) {
            ResourceSet rs = (ResourceSet)other;
            if(data == null) {
                return !rs.contains(sizeOrData);
            } else {
                for(int i = 0;i < sizeOrData ; i++) 
                    if(rs.contains(data[i])) return false;
                return true;
            }
        } else {
            if(data == null) {
                return !other.contains(sizeOrData);
            } else {
                for(int i = 0;i < sizeOrData ; i++) 
                    if(other.contains(data[i])) return false;
                return true;
            }
        }
    }

}