/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.impl.query;

import java.util.Collection;
import java.util.concurrent.Semaphore;

import org.simantics.db.common.exception.DebugException;
import org.simantics.db.impl.graph.ReadGraphImpl;
import org.simantics.db.procedure.ListenerBase;
import org.simantics.db.request.RequestFlags;

final public class DirectObjects extends CollectionBinaryQuery<IntProcedure> {

	private DirectObjects(final int r1, final int r2) {
		super(r1, r2);
	}

	@Override
	public int type() {
		return RequestFlags.INVALIDATE;
	}
	
    @Override
    public void clearResult(QuerySupport support) {
    	setResult(INVALID_RESULT);
    }

	final static DirectObjects entry(final QueryProcessor provider, final int r1, final int r2) {

		return (DirectObjects)provider.directObjectsMap.get(id(r1,r2));

	}
	
	final static Collection<DirectObjects> entries(final QueryProcessor processor, final int r1) {
		DoubleKeyQueryHashMap<IntProcedure> hash = processor.directObjectsMap;
		return hash.values(r1);
	}

	final static void runner(ReadGraphImpl graph, final int r1, final int r2, CacheEntry parent, final ListenerBase listener, final IntProcedure procedure) {

    	QueryProcessor processor = graph.processor;
		
		DirectObjects entry = (DirectObjects)processor.directObjectsMap.get(id(r1,r2));
		if(entry == null) {
			
        	entry = new DirectObjects(r1, r2);
        	entry.setPending();
        	entry.clearResult(processor.querySupport);
        	entry.putEntry(processor);
        	
			processor.performForEach(graph, entry, parent, listener, procedure);
			
		} else {
			
            if(entry.isPending()) {
                synchronized(entry) {
                    if(entry.isPending()) {
                        processor.registerDependencies(graph, entry, parent, listener, procedure, false);
						entry.computeForEach(graph, processor, procedure, true);
                        return;
                    }
                }
            }
        	
			processor.performForEach(graph, entry, parent, listener, procedure);
			
		}

	}

	final public static void queryEach(ReadGraphImpl graph, final int r1, final int r2, final QueryProcessor provider, final CacheEntry parent, final ListenerBase listener, final IntProcedure procedure) {

        assert(r1 != 0);
        assert(r2 != 0);
        
        if(parent == null && listener == null) {
        	DirectObjects.computeForEach(graph, r1, r2, null, procedure);
        } else {
        	runner(graph, r1, r2, parent, listener, procedure);
        }

	}

	@Override
	public BinaryQuery<IntProcedure> getEntry(QueryProcessor provider) {
		return provider.directObjectsMap.get(id);
	}

	@Override
	public void putEntry(QueryProcessor provider) {
		provider.directObjectsMap.put(id, this);
	}

	@Override
	final public void removeEntry(QueryProcessor provider) {
		provider.directObjectsMap.remove(id);
	}

	@Override
	public void computeForEach(ReadGraphImpl graph, final QueryProcessor queryProvider, final IntProcedure procedure, final boolean store) {
		computeForEach(graph, r1(), r2(), this, procedure);
	}

	static public void computeForEach(ReadGraphImpl graph, int r1, int r2, final DirectObjects entry, final IntProcedure procedure) {

		QueryProcessor processor = graph.processor;
		
		processor.querySupport.ensureLoaded(graph, r1, r2);

		processor.querySupport.getObjects(graph, r1, r2, new IntProcedure() {

			@Override
			public void execute(ReadGraphImpl graph, int i) {
				procedure.execute(graph, i);
			}

			@Override
			public void finished(ReadGraphImpl graph) {
			}

			@Override
			public void exception(ReadGraphImpl graph, Throwable t) {
				if(DebugException.DEBUG) new DebugException(t).printStackTrace();
			}

		});

		if(entry != null) entry.finish(graph, processor);
		procedure.finished(graph);

	}

	@Override
	public String toString() {
		return "DirectObjects[" + r1() + " - " + r2() + "]";
	}

	@Override
	public void setReady() {
		statusOrException = READY;
	}
	
	final private void finish(ReadGraphImpl graph, QueryProcessor provider) {
		setReady();
	}

	@Override
	public void performFromCache(ReadGraphImpl graph, QueryProcessor provider, IntProcedure procedure) {

		assert(isReady());
		computeForEach(graph, provider, procedure, false);

	}

	@Override
	public void recompute(ReadGraphImpl graph, QueryProcessor provider) {

        final Semaphore s = new Semaphore(0);
		
		computeForEach(graph, provider, new IntProcedure() {

			@Override
			public void finished(ReadGraphImpl graph) {
				s.release();
			}

			@Override
			public void exception(ReadGraphImpl graph, Throwable t) {
				throw new Error("Error in recompute.", t);
			}

			@Override
			public void execute(ReadGraphImpl graphd, int i) {
			}

		}, true);

        while(!s.tryAcquire()) {
        	provider.resume(graph);
        }

	}

    @Override
    boolean isImmutable(ReadGraphImpl graph) {
    	return graph.processor.isImmutable(r1());
    }
	
}

