/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
/*
 *
 * @author Toni Kalajainen
 */
package org.simantics.utils.datastructures.cache;

import java.lang.ref.WeakReference;
import java.util.HashMap;
import java.util.Map;

/**
 * WeakCachedProvider provides values and stores the results in a cache.
 * Cached values are held with weak references. Values are removed
 * automatically as they are disposed.
 *
 * Keys are referenced with strong references.
 *
 * @param <K> key type
 * @param <V> value type
 */
public class WeakCachedMapProvider<K, V> implements IMapProvider<K, V> {

	private Map<K, WeakReference<V>> cache = 
		new HashMap<K, WeakReference<V>>();
	
	private final IMapProvider<K, V> provider;
	
	/**
	 * Constructs new weak cache.
	 * 
	 * @param provider provider of values 
	 */
	public WeakCachedMapProvider(IMapProvider<K, V> provider)
	{
		assert(provider!=null);
		this.provider = provider;
	}	
	
	@Override
	public synchronized V get(K key) 
	{
		WeakReference<V> ref = cache.get(key);
		if (ref!=null) {
			V result = ref.get();
			if (result!=null) return result;
			cache.remove(ref);
		}
		
		V value = provider.get(key);
		assert(value!=null);
		
		ref = new WeakReference<V>(value);
		cache.put(key, ref);
		return value;		
	}
	
	/**
	 * Release all weak references
	 */
	public synchronized void clear()
	{
		cache.clear();
	}	

}
