/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.trend;

import java.awt.Color;
import java.util.concurrent.Executor;

import org.simantics.g2d.canvas.Hints;
import org.simantics.g2d.canvas.ICanvasContext;
import org.simantics.g2d.canvas.impl.CanvasContext;
import org.simantics.g2d.participant.BackgroundPainter;
import org.simantics.g2d.participant.CanvasBoundsParticipant;
import org.simantics.g2d.participant.CanvasGrab;
import org.simantics.g2d.participant.KeyToCommand;
import org.simantics.g2d.participant.KeyUtil;
import org.simantics.g2d.participant.MouseUtil;
import org.simantics.g2d.participant.PanZoomRotateHandler;
import org.simantics.g2d.participant.PointerPainter;
import org.simantics.g2d.participant.RulerPainter;
import org.simantics.g2d.participant.SymbolUtil;
import org.simantics.g2d.participant.TimeParticipant;
import org.simantics.g2d.participant.TransformUtil;
import org.simantics.g2d.tooltip.TooltipParticipant;
import org.simantics.history.Collector;
import org.simantics.history.HistoryManager;
import org.simantics.scenegraph.g2d.events.command.CommandKeyBinding;
import org.simantics.simulation.data.Datasource;
import org.simantics.simulation.data.Datasource.DatasourceListener;
import org.simantics.trend.configuration.TrendSpec;
import org.simantics.trend.impl.TrendNode;
import org.simantics.trend.impl.TrendParticipant;
import org.simantics.utils.datastructures.disposable.IDisposable;
import org.simantics.utils.datastructures.disposable.IDisposeListener;
import org.simantics.utils.datastructures.hints.IHintContext;
import org.simantics.utils.threads.IThreadWorkQueue;

public class TrendInitializer {

    public static CanvasContext defaultInitializeCanvas(CanvasContext cvsCtx, HistoryManager historian, Collector collector, final Datasource simulation, TrendSpec trendSpec)
    {
        // Create canvas context and a layer of interactors
        IHintContext h = cvsCtx.getDefaultHintContext();

        h.setHint(PanZoomRotateHandler.KEY_ADAPT_VIEWPORT_TO_RESIZED_CONTROL, false);
        h.setHint(PanZoomRotateHandler.KEY_DISABLE_PAN, true);
        h.setHint(PanZoomRotateHandler.KEY_DISABLE_ZOOM, true);
        cvsCtx.add( new PanZoomRotateHandler( false ) ); // Must be before TransformUtil

        // Support & Util Participants
        cvsCtx.add( new TransformUtil() );
        cvsCtx.add( new MouseUtil() );
        cvsCtx.add( new KeyUtil() );
        cvsCtx.add( new CanvasGrab() );
        cvsCtx.add( new SymbolUtil() );
        cvsCtx.add( new TimeParticipant() );
        cvsCtx.add( new CanvasBoundsParticipant() );

        // Add trend node
        TrendNode trendNode = cvsCtx.getSceneGraph().addNode(TrendNode.class);
        trendNode.setHistorian( historian, collector );
        trendNode.setTrendSpec( trendSpec );

        // Add trend participant/interactor
        TrendParticipant tp = new TrendParticipant();
        tp.setTrend( trendNode );
        cvsCtx.add( tp );

        if (simulation != null)
        	attachStepListener(simulation, cvsCtx, tp);

        // Debug participant(s)
        //canvasContext.add( new PointerPainter() );
//        canvasContext.add( new HandPainter() );
        h.setHint(PointerPainter.KEY_PAINT_POINTER, true);

        // Pan & Zoom & Rotate
        //canvasContext.add( new MousePanZoomInteractor() );
        //canvasContext.add( new MultitouchPanZoomRotateInteractor() );
        //canvasContext.add( new OrientationRestorer() );

        // Grid & Ruler & Background
        //cvsCtx.add( new GridPainter() );
        //cvsCtx.add( new RulerPainter() );
        cvsCtx.add( new BackgroundPainter() );

        h.setHint(Hints.KEY_GRID_COLOR, new Color(0.95f, 0.95f, 0.95f));
        h.setHint(Hints.KEY_BACKGROUND_COLOR, Color.WHITE);
        h.setHint(RulerPainter.KEY_RULER_BACKGROUND_COLOR, new Color(0.9f, 0.9f, 0.9f, 0.75f));
        h.setHint(RulerPainter.KEY_RULER_TEXT_COLOR, Color.BLACK);
        h.setHint(TimeParticipant.KEY_TIME_PULSE_INTERVAL, 50L);
        h.setHint(TimeParticipant.KEY_TIMER_ENABLED, true);
        h.setHint(TrendParticipant.KEY_TREND_DRAW_INTERVAL, 100L);

        // Key bindings
        cvsCtx.add( new KeyToCommand( CommandKeyBinding.DEFAULT_BINDINGS ) );

        ////// Diagram Participants //////

        cvsCtx.add( new TooltipParticipant());

        h.setHint(Hints.KEY_TOOL, Hints.POINTERTOOL);

        //NodeUtil.printSceneGraph( cvsCtx.getSceneGraph() );

        cvsCtx.assertParticipantDependencies();

        return cvsCtx;
    }
    
    public static TrendNode getTrendNode(CanvasContext ctx)
    {
    	TrendParticipant tp = ctx.getSingleItem( TrendParticipant.class );
    	return tp.getTrend();
    }

    public static void attachStepListener(final Datasource datasource, ICanvasContext context, TrendParticipant tp) {
        // Add simulation listener - Note listener must be removed when trend is closed
        final DatasourceListener stepListener = new StepListener( tp );
        datasource.addListener( stepListener );
        context.addDisposeListener(new IDisposeListener() {
            @Override
            public void onDisposed(IDisposable sender) {
                datasource.removeListener(stepListener);
            }
        });
    }

    /**
     *
     */
    public static class StepListener implements DatasourceListener {
        private final TrendParticipant trendParticipant;

        public StepListener(TrendParticipant trendParticipant) {
            this.trendParticipant = trendParticipant;
        }

        @Override
        public void onStep(Datasource source) {
            TrendNode node = trendParticipant.getTrend();
            if (!node.allPast()) node.datadirty = true;
        }

        @Override
        public Executor getExecutor() {
            return null;
        }
    }

	public static CanvasContext createDefaultCanvas(IThreadWorkQueue threadAccess, HistoryManager historian, Collector collector, Datasource simulation, TrendSpec trendSpec) {
		return defaultInitializeCanvas(new CanvasContext(threadAccess), historian, collector, simulation, trendSpec);
	}

}
