/*******************************************************************************
 * Copyright (c) 2007, 2013 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.objmap.structural.rules.domain;

import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.objmap.exceptions.MappingException;
import org.simantics.objmap.graph.rules.domain.IDomainAccessor;
import org.simantics.objmap.structural.StructuralResource;


public class StructuralRelatedObjectAccessor implements IDomainAccessor<StructuralResource,StructuralResource> {

    static Logger LOGGER = Logger.getLogger("org.simantics.objmap");
    
	Resource relation;
	boolean useTypeResource;
	
	private boolean preventStructuralChanges = true;
	private boolean preventStructuralRootChanges = true;
	
	public StructuralRelatedObjectAccessor(Resource relation, boolean useTypeResource) {
		this.relation = relation;
		this.useTypeResource = useTypeResource;
	}
	
	public StructuralRelatedObjectAccessor(Resource relation, boolean useTypeResource, boolean preventStructuralChanges, boolean preventStructuralRootChanges) {
		this.relation = relation;
		this.useTypeResource = useTypeResource;
		this.preventStructuralChanges = preventStructuralChanges;
		this.preventStructuralRootChanges = preventStructuralRootChanges;
	}
	
 private boolean preventChange(StructuralResource element) {
    	return preventStructuralChanges && element.isStructural() && (!element.isStructuralRoot()||preventStructuralRootChanges);	
    }

	@Override
	public StructuralResource get(ReadGraph g, StructuralResource element) throws MappingException {
		try {
			
			LOGGER.info("        RelatedObjectAccessor.get");
			
			if (!element.isStructural())
				return null;
			Resource instance = StructuralUtils.getContainingInstance(element);
			    
			Resource publicRelation = StructuralUtils.getPublishedRelation(g, element, relation);

		     if (publicRelation == null)
		    	return null;
			Resource r =  g.getPossibleObject(instance, publicRelation);
			if (r == null)
				return null;
			List<Resource> context = new ArrayList<Resource>();
			for (int i = 0; i < element.getContext().size()-1; i++)
				context.add(element.getContext().get(i));
			if (StructuralUtils.isStructuralInstance(g, r)) {
				return new StructuralResource(g, r, context,r);
			} else {
				return new StructuralResource(g, r, context);
			}
		} catch (DatabaseException e) {
			throw new MappingException(e);
		}
	}
	
	@Override
	public boolean set(WriteGraph g, StructuralResource element, StructuralResource value)
			throws MappingException {
		try {
		    LOGGER.info("        RelatedObjectAccessor.set");
		    Resource instance = StructuralUtils.getContainingInstance(element);
		    Resource publicRelation = null;
		    if (instance == null)
		    	return false;
		    publicRelation = StructuralUtils.getPublishedRelation(g, element, relation);
		    if (value == null) {
		    	if (publicRelation == null)
		    		return false;
		    	if (preventChange(element))
	    			return false;
		    	g.deny(instance, publicRelation);
		    	return true;
		    } else {
		    	if (publicRelation == null) {
		    		if (preventChange(element))
		    			return false;
		    		publicRelation = StructuralUtils.getOrCreatePublishedRelation(g, element, relation);
		    		g.claim(instance, publicRelation, value.getResource());
		    		return true;
		    	} else {
		    		Resource r = g.getPossibleObject(instance, publicRelation);
		    		if (r == null) {
		    			if (preventChange(element))
			    			return false;
		    			g.claim(instance, publicRelation, value.getResource());
		    			return true;
		    		} else {
		    			if (r.equals(value.getResource()))
		    				return false;
		    			if (preventChange(element))
			    			return false;
				    	g.deny(instance, publicRelation);
		    			g.claim(instance, publicRelation, value.getResource());
		    			return true;
		    		}
		    	}
		    }
		} catch (DatabaseException e) {
			throw new MappingException(e);
		}
		
	}

}
