/*******************************************************************************
 * Copyright (c) 2007, 2013 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.objmap.structural.rules.domain;

import java.util.Arrays;

import org.apache.log4j.Logger;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Statement;
import org.simantics.db.WriteGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.layer0.Layer0;
import org.simantics.objmap.exceptions.MappingException;
import org.simantics.objmap.graph.rules.domain.IDomainAccessor;
import org.simantics.objmap.structural.StructuralResource;


public class RelatedValueAccessor implements IDomainAccessor<StructuralResource,Object> {

    static Logger LOGGER = Logger.getLogger("org.simantics.objmap");
    
	Resource relation;
	Resource valueType;
	boolean useTypeResource;
	
	private boolean preventStructuralChanges = true;
	private boolean preventStructuralRootChanges = true;
	
	public RelatedValueAccessor(Resource relation, Resource valueType, boolean useTypeResource) {
		this.relation = relation;
		this.valueType = valueType;
		this.useTypeResource = useTypeResource;
	}
	
	public RelatedValueAccessor(Resource relation, Resource valueType, boolean useTypeResource, boolean preventStructuralChanges, boolean preventStructuralRootChanges) {
		this.relation = relation;
		this.valueType = valueType;
		this.useTypeResource = useTypeResource;
		this.preventStructuralChanges = preventStructuralChanges;
		this.preventStructuralRootChanges = preventStructuralRootChanges;
	}
	
	private boolean preventChange(StructuralResource element) {
    	return preventStructuralChanges && element.isStructural() && (!element.isStructuralRoot()||preventStructuralRootChanges);	
    }

	@Override
	public Object get(ReadGraph g, StructuralResource element) throws MappingException {
		try {
		    LOGGER.info("        RelatedValueAccessor.get");
		    Resource res = getServiceResource(g, element);
		    if (res == null)
		    	return null;
			Resource valueResource = g.getPossibleObject(res, relation);
			if(valueResource == null)
				return null;
			return g.getValue(valueResource);
		} catch (DatabaseException e) {
			throw new MappingException(e);
		}
	}
	
	@Override
	public boolean set(WriteGraph g, StructuralResource relement, Object value)
			throws MappingException {
		try {
		    LOGGER.info("        RelatedValueAccessor.set");
		    
		    Resource element = getServiceResource(g, relement);
		    if (element == null)
		    	return false;
		    Statement valueStatement = g.getPossibleStatement(element, relation);
			if(valueStatement == null) {
				if(value == null)
					return false;
				if (preventChange(relement))
					return false;
				Resource valueResource = g.newResource();
				g.claim(valueResource, Layer0.getInstance(g).InstanceOf, null,
						valueType);
				g.claim(element, relation, valueResource);
				g.claimValue(valueResource, value);				
				return true;
			}
			else {
				if(value == null) {
					if (preventChange(relement))
						return false;
					if (!valueStatement.isAsserted(element)) {
						g.deny(valueStatement.getObject());
						return true;
					} else {
						return false;
					}
					
				}				
				Object currentValue = g.getValue(valueStatement.getObject());
				if(equals(currentValue,value))
					return false;
				if (preventChange(relement))
					return false;
				if (!valueStatement.isAsserted(element))
					g.claimValue(valueStatement.getObject(), value);
				else {
					Resource valueResource = g.newResource();
					g.claim(valueResource, Layer0.getInstance(g).InstanceOf, null,
							valueType);
					g.claim(element, relation, valueResource);
					g.claimValue(valueResource, value);
				}
				return true;
			}
		} catch (DatabaseException e) {
			throw new MappingException(e);
		}
		
	}
	
	private Resource getServiceResource(ReadGraph g, StructuralResource element) {
		if (!useTypeResource)
			return element.getResource();
		return element.getTypeResource();
	}
	
	private boolean equals(Object o1, Object o2) {
		if (o1 instanceof boolean[])
			Arrays.equals((boolean[])o1,(boolean[])o2);
		if (o1 instanceof int[])
			Arrays.equals((int[])o1,(int[])o2);
		if (o1 instanceof float[])
			Arrays.equals((float[])o1,(float[])o2);
		if (o1 instanceof double[])
			Arrays.equals((double[])o1,(double[])o2);
		if (o1 instanceof byte[])
			Arrays.equals((byte[])o1,(byte[])o2);
		return o1.equals(o2);
		
	}

}
