/*******************************************************************************
 * Copyright (c) 2007, 2013 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.objmap.structural.rules.domain;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import org.apache.log4j.Logger;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.objmap.exceptions.MappingException;
import org.simantics.objmap.graph.rules.domain.IDomainAccessor;
import org.simantics.objmap.graph.rules.domain.MappingUtils;
import org.simantics.objmap.structural.StructuralResource;


public class RelatedObjectsAccessor implements IDomainAccessor<StructuralResource,Collection<StructuralResource>> {

    static Logger LOGGER = Logger.getLogger("org.simantics.objmap");
    
	Resource relation;
	boolean deleteExtraObjects;
	boolean useTypeResource;
	
	private boolean preventStructuralChanges = true;
	private boolean preventStructuralRootChanges = true;

	public RelatedObjectsAccessor(Resource relation, boolean deleteExtraObjects, boolean useTypeResource) {
        super();
        this.relation = relation;
        this.deleteExtraObjects = deleteExtraObjects;
        this.useTypeResource = useTypeResource;
    }
	
	public RelatedObjectsAccessor(Resource relation, boolean deleteExtraObjects, boolean useTypeResource, boolean preventStructuralChanges, boolean preventStructuralRootChanges) {
        super();
        this.relation = relation;
        this.deleteExtraObjects = deleteExtraObjects;
        this.useTypeResource = useTypeResource;
		this.preventStructuralChanges = preventStructuralChanges;
		this.preventStructuralRootChanges = preventStructuralRootChanges;
    }

    @Override
	public Collection<StructuralResource> get(ReadGraph g, StructuralResource element) throws MappingException {
		try {
		    LOGGER.info("        RelatedObjectsAccessor.get");
			
		   
		    Resource res = getServiceResource(g, element);
		    if (res == null)
		    	return Collections.emptyList();
		    Collection<Resource> coll = g.getObjects(res, relation);
			List<StructuralResource> result = new ArrayList<StructuralResource>(coll.size());
			for (Resource r : coll) {
				if (StructuralUtils.isStructuralInstance(g, r)) {
					result.add(new StructuralResource(g, r, element.getContext(),r));
				} else {
					result.add(new StructuralResource(g, r, element.getContext()));
				}
			}
			return result;
		} catch (DatabaseException e) {
			throw new MappingException(e);
		}
	}
    
    private boolean preventChange(StructuralResource element) {
    	return preventStructuralChanges && element.isStructural() && (!element.isStructuralRoot()||preventStructuralRootChanges);	
    }
	
	@Override
	public boolean set(WriteGraph g, StructuralResource element, Collection<StructuralResource> value)
			throws MappingException {
		try {
		    LOGGER.info("        RelatedObjectsAccessor.set");
		    Resource res = getServiceResource(g, element);
		    if (res == null)
		    	return false;
		    if (preventChange(element))
				return false;
		    Resource[] arr = new Resource[value.size()];
		    int i = 0; 
		    for (StructuralResource sr : value) {
		    	arr[i++] = sr.getResource();
		    }
			return MappingUtils.synchronizeStatements(g, res, relation, 
			        arr, deleteExtraObjects);
		} catch (DatabaseException e) {
			throw new MappingException(e);
		}
		
	}

	private Resource getServiceResource(ReadGraph g, StructuralResource element) {
		if (!useTypeResource)
			return element.getResource();
		return element.getTypeResource();
	}
}
