/*******************************************************************************
 * Copyright (c) 2007, 2013 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.objmap.graph.rules;

import java.util.ArrayList;
import java.util.Collection;

import org.apache.log4j.Logger;
import org.simantics.db.ReadGraph;
import org.simantics.db.WriteGraph;
import org.simantics.objmap.backward.IBackwardMapping;
import org.simantics.objmap.bidirectional.IBidirectionalMappingRule;
import org.simantics.objmap.exceptions.MappingException;
import org.simantics.objmap.forward.IForwardMapping;
import org.simantics.objmap.graph.rules.domain.IDomainAccessor;
import org.simantics.objmap.graph.rules.range.IRangeAccessor;


/**
 * A rule that synchronizes collection of elements between
 * domain and range accessors. Elements are mapped from
 * between domain and range during the synchronization.
 * @author Hannu Niemist
 */
public class MappedElementsRule<Domain, Range> implements IBidirectionalMappingRule<Domain, Range> {

    static Logger LOGGER = Logger.getLogger("org.simantics.objmap");

    IDomainAccessor<Domain,Collection<Domain>> domainAccessor;
    IRangeAccessor<Range,Collection<Range>> rangeAccessor;

    public MappedElementsRule(IDomainAccessor<Domain,Collection<Domain>> domainAccessor,
            IRangeAccessor<Range,Collection<Range>> rangeAccessor) {
        this.domainAccessor = domainAccessor;
        this.rangeAccessor = rangeAccessor;
    }

    @Override
    public boolean updateDomain(WriteGraph g, IBackwardMapping<Domain, Range> map,
            Domain domainElement, Range rangeElement)
    throws MappingException {
        LOGGER.info("    MappedElementsRule.updateDomain");
        // Snapshot the accessed range value for concurrency safety.
        // NOTE: still assumes that the accessed collection is concurrent or
        // synchronized for toArray to be atomic.
        Collection<Range> value = rangeAccessor.get(rangeElement);
        Object[] rangeSnapshot = value.toArray();
        ArrayList<Domain> mappedValue = new ArrayList<Domain>(rangeSnapshot.length);
        for (Object obj : rangeSnapshot)
            mappedValue.add(map.inverseMap(g, (Range)obj));//map.inverseGet((Range)obj));
        return domainAccessor.set(g, domainElement, mappedValue);
    }

    @Override
    public boolean updateRange(ReadGraph g, IForwardMapping<Domain, Range> map,
            Domain domainElement, Range rangeElement)
    throws MappingException {
        LOGGER.info("    MappedElementsRule.updateRange");
        Collection<Domain> value = domainAccessor.get(g, domainElement);
        ArrayList<Range> mappedValue = new ArrayList<Range>(value.size());
        for(Domain r : value)
            mappedValue.add(map.map(g, r));//map.get(r));
        return rangeAccessor.set(rangeElement, mappedValue);
    }
    
    public void createDomain(WriteGraph g, IBackwardMapping<Domain,Range> map, Domain domainElement, Range rangeElement) throws MappingException {
    	updateDomain(g, map, domainElement, rangeElement);
    };
    
    public void createRange(ReadGraph g, IForwardMapping<Domain,Range> map, Domain domainElement, Range rangeElement) throws MappingException {
    	updateRange(g, map, domainElement, rangeElement);
    };
}
